//-----------------------------------------------------------------------------
// Microsoft OLE DB RowsetViewer
// Copyright (C) 1994 - 1998 By Microsoft Corporation.
//	  
// @doc
//												  
// @module CDIALOG.CPP
//
//-----------------------------------------------------------------------------------
						  

/////////////////////////////////////////////////////////////////
// Includes					 
//
/////////////////////////////////////////////////////////////////
#include "common.h"
#include "CDialog.h"
#include "Main.h"

#include <richedit.h>	//RichEdit Control Header

/////////////////////////////////////////////////////////////////////
// Defines
//
/////////////////////////////////////////////////////////////////////
enum PROPICONS
{
	IMAGE_NORMAL		= 0,
	IMAGE_READONLY		= 1,
	IMAGE_ERROR			= 2,
};

//PROPINFO
enum PROPINFO_COLUMNS
{
	PROPINFO_NAME, 
	PROPINFO_TYPE,
	PROPINFO_VALUE,
	PROPINFO_FLAGS, 
	PROPINFO_DESC,
};

//PROPERTY
enum PROP_COLUMNS
{
	PROP_NAME, 
	PROP_TYPE,
	PROP_VALUE,
	PROP_OPTIONS, 
	PROP_COLID,
	PROP_DESC,
};

enum STATEICONS
{
	STATE_NORMAL		= 0,
	STATE_CHECKED		= 1,
	STATE_UNCHECKED		= 2,
};

#define NOTSET -2

#define HKEY_ROWSETVIEWER  HKEY_CURRENT_USER
#define szCONFIG_KEY			"Software\\Microsoft\\OLE DB\\RowsetViewer\\Config"
#define szRECENTCONFIG_KEY		"Software\\Microsoft\\OLE DB\\RowsetViewer\\RecentConfig"
#define szRECENTFILE_KEY		"Software\\Microsoft\\OLE DB\\RowsetViewer\\RecentFile"
#define szRECENTINITSTRING_KEY	"Software\\Microsoft\\OLE DB\\RowsetViewer\\RecentInitString"
#define szOPTIONS_KEY			"Software\\Microsoft\\OLE DB\\RowsetViewer\\Options"

//JOLT specific schema rowsets
//Only included here so the provider specific guids have a name map rather than
//trying to indentify them by guid value
const GUID DBSCHEMA_JETOLEDB_REPLPARTIALFILTERLIST	= {0xe2082df0,0x54ac,0x11d1,{0xbd,0xbb,0x00,0xc0,0x4f,0xb9,0x26,0x75}};
const GUID DBSCHEMA_JETOLEDB_REPLCONFLICTTABLES		= {0xe2082df2,0x54ac,0x11d1,{0xbd,0xbb,0x00,0xc0,0x4f,0xb9,0x26,0x75}};
const GUID DBSCHEMA_JETOLEDB_USERROSTER				= {0x947bb102,0x5d43,0x11d1,{0xbd,0xbf,0x00,0xc0,0x4f,0xb9,0x26,0x75}};
const GUID DBSCHEMA_JETOLEDB_ISAMSTATS				= {0x8703b612,0x5d43,0x11d1,{0xbd,0xbf,0x00,0xc0,0x4f,0xb9,0x26,0x75}};


/////////////////////////////////////////////////////////////////////
// Schema Constraint Mapping and Information
//
/////////////////////////////////////////////////////////////////////
static const SCHEMAINFO g_rgSchemaInfo[] = 
{
	& VALUE_CHAR(DBSCHEMA_ASSERTIONS),				3, {1,	"CONSTRAINT_CATALOG",	2,	"CONSTRAINT_SCHEMA",	3,	"CONSTRAINT_NAME"},
	& VALUE_CHAR(DBSCHEMA_CATALOGS),				1, {1,	"CATALOG_NAME"},
	& VALUE_CHAR(DBSCHEMA_CHARACTER_SETS),			3, {1,	"CHARACTER_SET_CATALOG",2,	"CHARACTER_SET_SCHEMA", 3,	"CHARACTER_SET_NAME"},
	& VALUE_CHAR(DBSCHEMA_CHECK_CONSTRAINTS),		3, {1,	"CONSTRAINT_CATALOG",	2,	"CONSTRAINT_SCHEMA",	3,	"CONSTRAINT_NAME"},
	& VALUE_CHAR(DBSCHEMA_COLLATIONS),				3, {1,	"COLLATIONS_CATALOG",	2,	"COLLATIONS_SCHEMA",	3,	"COLLATIONS_NAME"},

	& VALUE_CHAR(DBSCHEMA_COLUMN_DOMAIN_USAGE),		4, {1,	"DOMAIN_CATALOG",		2,	"DOMAIN_SCHEMA",		3,	"DOMAIN_NAME",			4,	"COLUMN_NAME"},
	& VALUE_CHAR(DBSCHEMA_COLUMN_PRIVILEGES),		6, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME",			4,	"COLUMN_NAME",				5,	"GRANTOR",				6,	"GRANTEE"},
	& VALUE_CHAR(DBSCHEMA_COLUMNS),					4, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME",			4,	"COLUMN_NAME"},
	& VALUE_CHAR(DBSCHEMA_CONSTRAINT_COLUMN_USAGE),	4, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME",			4,	"COLUMN_NAME"},
	& VALUE_CHAR(DBSCHEMA_CONSTRAINT_TABLE_USAGE),	3, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME"},

	& VALUE_CHAR(DBSCHEMA_FOREIGN_KEYS),			6, {1,	"PK_TABLE_CATALOG",		2,	"PK_TABLE_SCHEMA",		3,	"PK_TABLE_NAME",		4,	"FK_TABLE_CATALOG",			5,	"FK_TABLE_SCHEMA",		6,	"FK_TABLE_NAME"},
	& VALUE_CHAR(DBSCHEMA_INDEXES),					5, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"INDEX_NAME",			4,	"TYPE",						5,	"TABLE_NAME"},
	& VALUE_CHAR(DBSCHEMA_KEY_COLUMN_USAGE),		7, {1,	"CONSTRAINT_CATALOG",	2,	"CONSTRAINT_SCHEMA",	3,	"CONSTRAINT_NAME",		4,	"TABLE_CATALOG",			5,	"TABLE_SCHEMA",			6,	"TABLE_NAME",		7,	"COLUMN_NAME"},
	& VALUE_CHAR(DBSCHEMA_PRIMARY_KEYS),			3, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME"},

	& VALUE_CHAR(DBSCHEMA_PROCEDURE_COLUMNS),		4, {1,	"PROCEDURE_CATALOG",	2,	"PROCEDURE_SCHEMA",		3,	"PROCEDURE_NAME",		4,	"COLUMN_NAME"},
	& VALUE_CHAR(DBSCHEMA_PROCEDURE_PARAMETERS),	4, {1,	"PROCEDURE_CATALOG",	2,	"PROCEDURE_SCHEMA",		3,	"PROCEDURE_NAME",		4,	"PARAMETER_NAME"},
	& VALUE_CHAR(DBSCHEMA_PROCEDURES),				4, {1,	"PROCEDURE_CATALOG",	2,	"PROCEDURE_SCHEMA",		3,	"PROCEDURE_NAME",		4,	"PROCEDURE_TYPE"},
	
	& VALUE_CHAR(DBSCHEMA_PROVIDER_TYPES),			2, {1,	"DATA_TYPE",			2,	"BEST_MATCH"},
	& VALUE_CHAR(DBSCHEMA_REFERENTIAL_CONSTRAINTS), 3, {1,	"CONSTRAINT_CATALOG",	2,	"CONSTRAINT_SCHEMA",	3,	"CONSTRAINT_NAME"},
	& VALUE_CHAR(DBSCHEMA_SCHEMATA),				3, {1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"SCHEMA_OWNER"},
	
	& VALUE_CHAR(DBSCHEMA_SQL_LANGUAGES),			0, {1,	""},
	& VALUE_CHAR(DBSCHEMA_STATISTICS),				3, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME"},

	& VALUE_CHAR(DBSCHEMA_TABLE_CONSTRAINTS),		7, {1,	"CONSTRAINT_CATALOG",	2,	"CONSTRAINT_SCHEMA",	3,	"CONSTRAINT_NAME",		4,	"TABLE_CATALOG",			5,	"TABLE_SCHEMA",			6,	"TABLE_NAME",		7,	"CONSTRAINT_TYPE"},
	& VALUE_CHAR(DBSCHEMA_TABLE_PRIVILEGES),		5, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME",			4,	"GRANTOR",					5,	"GRANTEE"},
	& VALUE_CHAR(DBSCHEMA_TABLES),					4, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			4,	"TABLE_NAME",			3,	"TABLE_TYPE"},

	& VALUE_CHAR(DBSCHEMA_TRANSLATIONS),			3, {1,	"TRANSLATION_CATALOG",	2,	"TRANSLATION_SCHEMA",	3,	"TRANSLATION_NAME"},
	& VALUE_CHAR(DBSCHEMA_USAGE_PRIVILEGES),		6, {1,	"OBJECT_CATALOG",		2,	"OBJECT_SCHEMA",		3,	"OBJECT_NAME",			4,	"OBJECT_TYPE",				5,	"GRANTOR",				6,	"GRANTEE"},

	& VALUE_CHAR(DBSCHEMA_VIEW_COLUMN_USAGE),		3, {1,	"VIEW_CATALOG",			2,	"VIEW_SCHEMA",			3,	"VIEW_NAME"},
	& VALUE_CHAR(DBSCHEMA_VIEW_TABLE_USAGE),		3, {1,	"VIEW_CATALOG",			2,	"VIEW_SCHEMA",			3,	"VIEW_NAME"},
	& VALUE_CHAR(DBSCHEMA_VIEWS),					3, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			3,	"TABLE_NAME"},

	//2.0
	& VALUE_CHAR(DBSCHEMA_TABLES_INFO),				4, {1,	"TABLE_CATALOG",		2,	"TABLE_SCHEMA",			4,	"TABLE_NAME",			3,	"TABLE_TYPE"},

	//OLAP Extensions
	& VALUE_CHAR(MDSCHEMA_CUBES),					3, {1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"CUBE_NAME"},
	& VALUE_CHAR(MDSCHEMA_DIMENSIONS),				5, {1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"CUBE_NAME",			4,	"DIMENSION_NAME",			5,	"DIMENSION_UNIQUE_NAME"},
	& VALUE_CHAR(MDSCHEMA_HIERARCHIES),				6, {1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"CUBE_NAME",			4,	"DIMENSION_UNIQUE_NAME",	5,	"HIERARCHY_NAME",		 6,	"HIERARCHY_UNIQUE_NAME"},
    & VALUE_CHAR(MDSCHEMA_LEVELS),					7, {1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"CUBE_NAME",			4,	"DIMENSION_UNIQUE_NAME",	5,	"HIERARCHY_UNIQUE_NAME", 6,	"LEVEL_NAME",		 7,	"LEVEL_UNIQUE_NAME"},
	& VALUE_CHAR(MDSCHEMA_MEASURES),				5, {1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"CUBE_NAME",			4,	"MEASURE_NAME",				5,	"MEASURE_UNIQUE_NAME"},
	& VALUE_CHAR(MDSCHEMA_PROPERTIES),				9, {1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"CUBE_NAME",			4,	"DIMENSION_UNIQUE_NAME",	5,	"HIERARCHY_UNIQUE_NAME", 6,	"LEVEL_UNIQUE_NAME", 7,	"MEMBER_UNIQUE_NAME",	8,	"PROPERTY_NAME",	9,	"PROPERTY_TYPE"},
	& VALUE_CHAR(MDSCHEMA_MEMBERS),					11,{1,	"CATALOG_NAME",			2,	"SCHEMA_NAME",			3,	"CUBE_NAME",			4,	"DIMENSION_UNIQUE_NAME",	5,	"HIERARCHY_UNIQUE_NAME", 6,	"LEVEL_UNIQUE_NAME", 7,	"LEVEL_NUMBER",			8,	"MEMBER_NAME",		9,	"MEMBER_UNIQUE_NAME",	10,	"MEMBER_TYPE",	11,	"MDTREE_OPERATOR"},

	//JOLT specific schema rowsets
	//Only included here so the provider specific guids have a name map rather than
	//trying to indentify them by guid value
	& VALUE_CHAR(DBSCHEMA_JETOLEDB_REPLPARTIALFILTERLIST),	0, {1,	""},
	& VALUE_CHAR(DBSCHEMA_JETOLEDB_REPLCONFLICTTABLES),		0, {1,	""},
	& VALUE_CHAR(DBSCHEMA_JETOLEDB_USERROSTER),				0, {1,	""},
	& VALUE_CHAR(DBSCHEMA_JETOLEDB_ISAMSTATS),				0, {1,	""},
};
			

/////////////////////////////////////////////////////////////////////
// Helper Functions
//
/////////////////////////////////////////////////////////////////////
void*  SetThis(HWND hWnd, LONG lParam)
{
	ASSERT(hWnd);
	SetWindowLong(hWnd, GWL_USERDATA, lParam);
	return (void*)lParam;
}

void*  GetThis(HWND hWnd)
{
	ASSERT(hWnd);
	return (void*)GetWindowLong(hWnd, GWL_USERDATA);
}


/////////////////////////////////////////////////////////////////////
// CDialogBase::CDialogBase
//
/////////////////////////////////////////////////////////////////////
CDialogBase::CDialogBase(HWND hWnd, HINSTANCE hInst)
{
	ASSERT(hInst);
	
	m_hWnd	= hWnd;
	m_hInst = hInst;
}

/////////////////////////////////////////////////////////////////////
// CDialogBase::~CDialogBase
//
/////////////////////////////////////////////////////////////////////
CDialogBase::~CDialogBase()
{
//	Destroy();
}


/////////////////////////////////////////////////////////////////////
// ULONG CDialogBase::Destroy
//
/////////////////////////////////////////////////////////////////////
ULONG CDialogBase::Destroy()
{
	if(m_hWnd)
	{
		EndDialog(m_hWnd, 0);
		m_hWnd = NULL;
	}

	return 0;
}


////////////////////////////////////////////////////////////////
// CListBox::CListBox
//
/////////////////////////////////////////////////////////////////
CListBox::CListBox(HWND hWnd, HINSTANCE hInst, CMainWindow* pCMainWindow)
	: CDialogBase(hWnd, hInst)
{
	//Data
	ASSERT(pCMainWindow);
	m_pCMainWindow = pCMainWindow;
	
	//Timeings
	m_timerValue.QuadPart	= 0;
	m_timerFreq.QuadPart	= 0;
	QueryPerformanceFrequency(&m_timerFreq);

	//Other
	m_iNotificationIndex = LB_APPEND;
}


////////////////////////////////////////////////////////////////
// CListBox::~CListBox
//
/////////////////////////////////////////////////////////////////
CListBox::~CListBox()
{
	//This window has already been removed, From the WM_DESTROY...
	m_hWnd			= NULL;
}


////////////////////////////////////////////////////////////////
// CListBox::Display
//
/////////////////////////////////////////////////////////////////
ULONG CListBox::Display(HWND hWndParent, UINT x, UINT y, UINT iWidth, UINT iHeight, DWORD dwFlags, void* lParam)
{
	//CreateListBox
	ASSERT(m_hWnd == NULL);
	m_hWnd = CreateWindowEx(
		 WS_EX_CLIENTEDGE,          // ex style
		 m_pCMainWindow->m_hLibRichEdit ? "RICHEDIT" : "EDIT",	// class name - defined in commctrl.h
         NULL,                      // window text
         dwFlags | WS_TABSTOP | WS_CHILD | WS_BORDER | WS_HSCROLL | WS_VSCROLL | ES_LEFT | ES_AUTOVSCROLL | ES_AUTOHSCROLL | ES_MULTILINE | ES_WANTRETURN | /*ES_NOHIDESEL |*/ ES_READONLY,
		 x,                         // x position
         y,                         // y position
         iWidth,		            // width
         iHeight,                   // height
		 hWndParent,                // parent
         (HMENU)IDC_LISTBOX,        // ID
         m_hInst,                   // instance
         lParam);                   // lParam
	
	SetThis(m_hWnd, (LPARAM)lParam);
	SendMessage(m_hWnd, EM_SETTEXTMODE , TM_PLAINTEXT | TM_MULTILEVELUNDO, 0);
	return (ULONG)m_hWnd;
}



////////////////////////////////////////////////////////////////
// CListBox::GetOptionsObj
//
/////////////////////////////////////////////////////////////////
COptionsDlg* CListBox::GetOptionsObj()
{
	ASSERT(m_pCMainWindow);
	ASSERT(m_pCMainWindow->m_pCOptionsDlg);
	return m_pCMainWindow->m_pCOptionsDlg;
}

	

////////////////////////////////////////////////////////////////
// CListBox::InitControls
//
/////////////////////////////////////////////////////////////////
BOOL CListBox::InitControls(HWND hWnd)
{
	m_hWnd = hWnd;
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CListBox::RefreshControls
//
/////////////////////////////////////////////////////////////////
BOOL CListBox::RefreshControls()
{
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CListBox::CopyTo
//
/////////////////////////////////////////////////////////////////
BOOL CListBox::CopyTo(HWND hWndTo)
{
	//No-op
	if(hWndTo == NULL)
		return FALSE;
	CHAR szBuffer[MAX_QUERY_LEN];

	//Move the Caret to the Begining
	SendMessage(hWndTo, EM_SETSEL, 0, 0);

	//Copy the contents of this ListBox to the destination...
	LONG iCount = SendMessage(m_hWnd, EM_GETLINECOUNT, 0, 0);
	for(LONG i=0; i<iCount; i++)
	{
		//Get the Text for this Line
		//EM_GETLINE assumes the first byte of the buffer
		//indicates the total size of the buffer
		((DWORD*)szBuffer)[0] = MAX_QUERY_LEN-1;
		LONG iChars = SendMessage(m_hWnd, EM_GETLINE, i, (LPARAM)szBuffer);

		//Supposedly EM_GETLINE doesn't contain a NULL Terminator?!
		ASSERT(iChars >=0 && iChars < MAX_QUERY_LEN);
		szBuffer[iChars] = EOL;

		if(szBuffer[0])
		{
			//Now Copy this text to the Destination
			SendMessage(hWndTo, EM_REPLACESEL, TRUE, (LPARAM)szBuffer);
			//Carriage return in EditControl is (2 carriage returns, 1 line feed)
			SendMessageA(hWndTo, EM_REPLACESEL, TRUE, (LPARAM)"\r\r\n");
		}
	}

	return TRUE;
}


//////////////////////////////////////////////////////////////////
// LONG CListBox::OutputLineEnd
//
//////////////////////////////////////////////////////////////////
LONG CListBox::OutputLineEnd()
{
	//Standard MultiLine Edit Controls require (\r\r\n)
	//Anything else results in no line break
	//Even RichEdit controls "\n" works but when copying to text files they are lost
	SendMessageA(m_hWnd, EM_REPLACESEL, TRUE, (LPARAM)"\r\r\n");
	return 0;
}


//////////////////////////////////////////////////////////////////
// LONG CListBox::OutputText
//
//////////////////////////////////////////////////////////////////
LONG CListBox::OutputText(LONG iIndex, CHAR* pszFmt, ...)
{
	ASSERT(pszFmt);

	//No-op - no output window...
	if(m_hWnd == NULL)
		return 0;

	va_list		marker;
	CHAR		szBuffer[MAX_QUERY_LEN];

	// Use format and arguements as input
	//This version will not overwrite the stack, since it only copies
	//upto the max size of the array
	va_start(marker, pszFmt);
	_vsnprintf(szBuffer, MAX_QUERY_LEN, pszFmt, marker);
	va_end(marker);

	//Make sure there is a NULL Terminator, vsnwprintf will not copy
	//the terminator if length==MAX_NAME_LEN
	szBuffer[MAX_QUERY_LEN-1] = EOL;	
	
	//Delegate 				  
	if(iIndex == LB_APPEND)
	{
		//Move the Caret to the End
		SendMessageA(m_hWnd, EM_SETSEL, LONG_MAX, LONG_MAX);
		//Save the Begining Char Location (so we can overwrite with Post Message)
		SendMessageA(m_hWnd, EM_GETSEL, (WPARAM)&iIndex, 0);
		//Append the New String
		SendMessageA(m_hWnd, EM_REPLACESEL, TRUE, (LPARAM)szBuffer);
		//Carriage return in EditControl is (2 carriage returns, 1 line feed)
		OutputLineEnd();
	}
	else
	{
		//Need to find the Length of the Saved Line
		LONG iLength = SendMessageA(m_hWnd, EM_LINELENGTH, iIndex, 0);
		//We need to Place the Post Message in the Saved Location...
		SendMessageA(m_hWnd, EM_SETSEL, iIndex, iIndex + iLength);
		//Now display the Post Message
		SendMessageA(m_hWnd, EM_REPLACESEL, TRUE, (LPARAM)szBuffer);
		//Move the Caret to the End
		SendMessageA(m_hWnd, EM_SETSEL, LONG_MAX, LONG_MAX);
		iIndex = LB_APPEND;
	}

	//Since we added a new item, scroll down to it...
//	SendMessage(m_hWnd, WM_VSCROLL,(WPARAM)SB_BOTTOM,(LPARAM)NULL);
	return iIndex;
}



//////////////////////////////////////////////////////////////////
// HRESULT CListBox::OutputNotification
//
//////////////////////////////////////////////////////////////////
HRESULT CListBox::OutputNotification(DWORD dwNotifyType, CHAR* pszFmt, ...)
{
	ASSERT(pszFmt);

	//Only if the user is interested in this message
	if(GetOptionsObj()->m_dwNotifyOpts & dwNotifyType)
	{
		va_list		marker;
		CHAR		szBuffer[MAX_QUERY_LEN];

		// Use format and arguements as input
		//This version will not overwrite the stack, since it only copies
		//upto the max size of the array
		va_start(marker, pszFmt);
		_vsnprintf(szBuffer, MAX_QUERY_LEN, pszFmt, marker);
		va_end(marker);

		//Make sure there is a NULL Terminator, vsnwprintf will not copy
		//the terminator if length==MAX_NAME_LEN
		szBuffer[MAX_QUERY_LEN-1] = EOL;	
		
		//Delegate 				  
		OutputText(LB_APPEND, "%s", szBuffer);
	}
	return S_OK;
}


//////////////////////////////////////////////////////////////////
// ULONG CListBox::OutputAddRef
//
//////////////////////////////////////////////////////////////////
ULONG CListBox::OutputAddRef(IUnknown* pIUnknown, CHAR* pszFmt)
{
	ULONG ulRefCount = 0;

	//AddRef
	if(pIUnknown)
	{
		ulRefCount = pIUnknown->AddRef();

		//Only if the user is interested in this message
		if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_ADDREF)
		{
			//Delegate 				  
			OutputText(LB_APPEND, "%s::AddRef() - %d", pszFmt ? pszFmt : "IUnknown", ulRefCount);
			m_iNotificationIndex = LB_APPEND;
		}
	}

	return ulRefCount;
}


//////////////////////////////////////////////////////////////////
// ULONG CListBox::OutputRelease
//
//////////////////////////////////////////////////////////////////
ULONG CListBox::OutputRelease(IUnknown** ppIUnknown, CHAR* pszFmt, ULONG ulExpectedRefCount)
{
	ASSERT(ppIUnknown);
	ULONG ulRefCount = 0;

	//Release
	if(*ppIUnknown)
	{
		//PreNotification
		//We actually nned to out a pre and a post for IUnknown::Release
		//Since IRowset::Release will send a notfication!  ROWSET_RELEASE
		if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_RELEASE)
			m_iNotificationIndex = OutputText(LB_APPEND, "%s::Release() - %d", pszFmt ? pszFmt : "IUnknown", ulRefCount);
		
		//Actual Release and NULL pointer
		ulRefCount = (*ppIUnknown)->Release();
		*ppIUnknown = NULL;

		//Only if the user is interested in this message
		if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_RELEASE)
		{
			//Delegate 				  
			OutputText(m_iNotificationIndex, "%s::Release() - %d", pszFmt ? pszFmt : "IUnknown", ulRefCount);
			m_iNotificationIndex = LB_APPEND;
		}

		//Display Errors if RefCount is not what is expected...
		if(ulExpectedRefCount==0 && ulRefCount!=0)
		{
			DisplayRefCountErrors(/*m_hWnd*/NULL, pszFmt, ulRefCount, ulExpectedRefCount);
		}
	}

	return ulRefCount;
}


//////////////////////////////////////////////////////////////////
// HRESULT CListBox::OutputQI
//
//////////////////////////////////////////////////////////////////
HRESULT CListBox::OutputQI(IUnknown* pIUnknown, REFIID riid, IUnknown** ppIUnknown, CHAR* pszFmt)
{
	ASSERT(ppIUnknown);
	HRESULT hr = E_NOINTERFACE;

	if(pIUnknown)
	{
		//QueryInterface
		hr = pIUnknown->QueryInterface(riid, (void**)ppIUnknown);

		//Only if the user is interested in this message
		if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_QUERYINTEFACE)
		{
			//Delegate 				  
			OutputText(LB_APPEND, "%s::QueryInterface(%s, &0x%08x) - %S", pszFmt ? pszFmt : "IUnknown", GetInterfaceName(riid), *ppIUnknown, GetErrorName(hr));
			m_iNotificationIndex = LB_APPEND;
		}
	}
	
	return hr;
}


//////////////////////////////////////////////////////////////////
// HRESULT CListBox::OutputPreMethod
//
//////////////////////////////////////////////////////////////////
HRESULT CListBox::OutputPreMethod(CHAR* pszFmt, ...)
{
	ASSERT(pszFmt);
	
	//Only if the user is interested in this message
	if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_METHODCALLS)
	{
		va_list		marker;
		CHAR		szBuffer[MAX_QUERY_LEN];

		// Use format and arguements as input
		//This version will not overwrite the stack, since it only copies
		//upto the max size of the array
		va_start(marker, pszFmt);
		_vsnprintf(szBuffer, MAX_QUERY_LEN, pszFmt, marker);
		va_end(marker);

		//Make sure there is a NULL Terminator, vsnwprintf will not copy
		//the terminator if length==MAX_NAME_LEN
		szBuffer[MAX_QUERY_LEN-1] = EOL;	
		
		//Delegate 				  
		m_iNotificationIndex = OutputText(LB_APPEND, "%s", szBuffer);
	}
	
	//Set Timings
	if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_TIMINGS)
	{
		if(!QueryPerformanceCounter(&m_timerValue))
			m_timerValue.QuadPart = 0;
	}
	return S_OK;
}


//////////////////////////////////////////////////////////////////
// HRESULT CListBox::OutputPostMethod
//
//////////////////////////////////////////////////////////////////
HRESULT CListBox::OutputPostMethod(HRESULT hrActual, CHAR* pszFmt, ...)
{
	ASSERT(pszFmt);
	
	//Only if the user is interested in this message
	if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_METHODCALLS)
	{
		va_list		marker;
		CHAR		szBuffer[MAX_QUERY_LEN];

		// Use format and arguements as input
		//This version will not overwrite the stack, since it only copies
		//upto the max size of the array
		va_start(marker, pszFmt);
		_vsnprintf(szBuffer, MAX_QUERY_LEN, pszFmt, marker);
		va_end(marker);

		//Make sure there is a NULL Terminator, vsnwprintf will not copy
		//the terminator if length==MAX_NAME_LEN
		szBuffer[MAX_QUERY_LEN-1] = EOL;	
		
		//Delegate 				  
		if(GetOptionsObj()->m_dwNotifyOpts & NOTIFY_TIMINGS && m_timerValue.QuadPart)
		{
			//Time Calculation
			double dTime = 0;
			LARGE_INTEGER timerValue;
			if(QueryPerformanceCounter(&timerValue))
				dTime = (double)(timerValue.QuadPart - m_timerValue.QuadPart) / (double)m_timerFreq.QuadPart;
			
			//Output Notiifcation
			OutputText(m_iNotificationIndex, "%s - %S - %.06f seconds", szBuffer, GetErrorName(hrActual), dTime);

			//Reset Timings
			m_timerValue.QuadPart = 0;
		}
		else
		{
			OutputText(m_iNotificationIndex, "%s - %S", szBuffer, GetErrorName(hrActual));
		}
		
		m_iNotificationIndex = LB_APPEND;
	}

	return hrActual;
}


////////////////////////////////////////////////////////////////
// CListBox::ClearNotifications
//
/////////////////////////////////////////////////////////////////
HRESULT CListBox::ClearNotifications()
{
	//Select all the Text and Cut it...
	//This method allows the text to be undone...
//	SendMessage(m_hWnd, EM_SETSEL, 0, -1);
//	SendMessage(m_hWnd, WM_CUT, 0, 0);

	//Since this window is ReadOnly, we can't Cut it, we can only delete...
	SendMessage(m_hWnd, WM_SETTEXT, 0, (LPARAM)"");
	m_pCMainWindow->m_fWarnMaxEditBuffer = TRUE;
	return S_OK;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::CConnectDlg
//
/////////////////////////////////////////////////////////////////
CConnectDlg::CConnectDlg(HWND hWnd, HINSTANCE hInst, CMainWindow* pCMainWindow)
	: CDialogBase(hWnd, hInst)
{
	ASSERT(pCMainWindow);
	CHAR szBuffer[MAX_NAME_LEN+1];

	//Data
	m_pCMainWindow	= pCMainWindow;
	m_pCListBox		= new CListBox(NULL, hInst, pCMainWindow);
	m_pCEnum		= new CEnum(NULL, pCMainWindow, m_pCListBox);
	m_pCPropDlg		= new CPropDlg(NULL, hInst, pCMainWindow, m_pCListBox);

	//Create CListBox window, (hidden - not WS_VISIBILE)
	m_pCListBox->Display(m_pCMainWindow->m_hWnd, 0, 0, 75, 75, WS_BORDER, this);

	//Enum
	m_pIUnknown				= NULL;
	m_pIParseDisplayName	= NULL;
	memset(&m_EnumInfo, 0, sizeof(ENUMINFO));
	wcscpy(m_EnumInfo.wszName, L"MSDASQL");

	//Properties
	m_cPropSets			= 0;
	m_rgPropSets		= NULL;

	m_hWndProvider		= NULL;
	m_hWndProperties	= NULL;
	m_hWndSecurity		= NULL;
	m_hWndOptions		= NULL;
	m_hWndTrace			= NULL;

	//ServiceComponents
	m_pIDataInitialize		= NULL;
	m_pIDBPromptInitialize	= NULL;

	//Reset all Defaults
	ResetDefaults();

	//Load Configuration Name, so we know which configuration is the default
	GetRegEntry(HKEY_ROWSETVIEWER, szCONFIG_KEY, "DefaultConfig",	m_szConfigName, MAX_NAME_LEN);

	//Load Recent Configurations
	for(ULONG i=0; i<MAX_RECENTCONFIGS; i++)
	{
		if(FAILED(GetRegEnumValue(HKEY_ROWSETVIEWER, szRECENTCONFIG_KEY, i, szBuffer, MAX_NAME_LEN)))
			break;
		
		m_listConfigs.AddTail(strDuplicate(szBuffer));
	}

	//Load Recent File
	for(i=0; i<MAX_RECENTFILES; i++)
	{
		if(FAILED(GetRegEnumValue(HKEY_ROWSETVIEWER, szRECENTFILE_KEY, i, szBuffer, MAX_NAME_LEN)))
			break;
		
		m_listFiles.AddTail(strDuplicate(szBuffer));
	}

	//Load the Values for this Configuration
	LoadDefaults();
}


////////////////////////////////////////////////////////////////
// CConnectDlg::~CConnectDlg
//
/////////////////////////////////////////////////////////////////
CConnectDlg::~CConnectDlg()
{
	//The MDIChild takes care of cleaning up itself and the DataSource
	SAFE_DELETE(m_pCEnum);
	SAFE_DELETE(m_pCPropDlg);
	SAFE_DELETE(m_pCListBox);

	SAFE_RELEASE(m_pIUnknown);
	SAFE_RELEASE(m_pIParseDisplayName);
	FreeProperties(&m_cPropSets, &m_rgPropSets);

	//Release ServiceComponents
	SAFE_RELEASE(m_pIDataInitialize);
	SAFE_RELEASE(m_pIDBPromptInitialize);

	//Save Recent Configs
	DelRegEntry(HKEY_ROWSETVIEWER, szRECENTCONFIG_KEY);
	while(!m_listConfigs.IsEmpty())
	{
		CHAR* pszConfigName = m_listConfigs.RemoveHead();
		if(FAILED(SetRegEntry(HKEY_ROWSETVIEWER, szRECENTCONFIG_KEY, pszConfigName, MAX_NAME_LEN)))
			break;

		SAFE_FREE(pszConfigName);
	}

	//Save Recent Files
	DelRegEntry(HKEY_ROWSETVIEWER, szRECENTFILE_KEY);
	while(!m_listFiles.IsEmpty())
	{
		CHAR* pszFileName = m_listFiles.RemoveHead();
		if(FAILED(SetRegEntry(HKEY_ROWSETVIEWER, szRECENTFILE_KEY, pszFileName, MAX_NAME_LEN)))
			break;

		SAFE_FREE(pszFileName);
	}

	//ServiceComponents
	CHAR szBuffer[MAX_QUERY_LEN];
	ConvertToMBCS(m_wszInitString, szBuffer, MAX_QUERY_LEN);
	DelRegEntry(HKEY_ROWSETVIEWER, szRECENTINITSTRING_KEY);
	SetRegEntry(HKEY_ROWSETVIEWER, szRECENTINITSTRING_KEY,	szBuffer, MAX_QUERY_LEN);
}


////////////////////////////////////////////////////////////////
// CConnectDlg::GetOptionsObj
//
/////////////////////////////////////////////////////////////////
COptionsDlg* CConnectDlg::GetOptionsObj()
{
	ASSERT(m_pCMainWindow);
	ASSERT(m_pCMainWindow->m_pCOptionsDlg);
	return m_pCMainWindow->m_pCOptionsDlg;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::ResetDefaults
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::ResetDefaults()
{
	//Configuration Name
	strcpy(m_szConfigName, "(Default)");
	
	//Provider Page
	m_wszLocation[0]	= wEOL;			//DBPROP_INIT_LOCATION
	m_wszDataSource[0]	= wEOL;			//DBPROP_INIT_DATASOURCE
	m_wszUserID[0]		= wEOL;			//DBPROP_AUTH_USERID
	m_wszPassword[0]	= wEOL;			//DBPROP_AUTH_PASSWORD
	m_dwPromptProp		= DBPROMPT_COMPLETE;
	m_dwhWndProp		= (LONG)m_pCMainWindow->m_hWnd;

	//Properties Page
	m_wszProvString[0]		= wEOL;		//DBPROP_INIT_PROVIDERSTRING
	m_wszCatalog[0]			= wEOL;		//DBPROP_INIT_CATALOG
	m_dwlcidProp			= NOTSET;

	m_dwAsynchProp			= NOTSET;
	m_dwTimeoutProp			= NOTSET;
	m_dwModeProp			= NOTSET;

	//Security Page
	m_dwProtectionProp		= NOTSET;
	m_dwImpersonateProp		= NOTSET;
	m_dwMaskPasswordProp	= NOTSET;
	m_dwEncryptPasswordProp	= NOTSET;
	m_dwCacheProp			= NOTSET;	
	m_dwPersistProp			= NOTSET;		
	m_dwPersistEncryptProp	= NOTSET;
	m_wszIntegrated[0]		= wEOL;		// DBPROP_INIT_INTEGRATED

	//Options Page
	m_dwCLSCTX				= CLSCTX_INPROC_SERVER;
	m_wszRemoteServer[0]	= wEOL;
	m_dwConnectOpts			= CONNECT_INITIALIZE | CONNECT_SETPROPERTIES | CONNECT_CREATESESSION | CONNECT_CREATECOMMAND;
	
	//Other Saved Config Info
	m_wszCmdBuffer[0]	= wEOL;

	//ServicecComponents
	m_wszInitString[0]  = wEOL;
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::LoadDefaults
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::LoadDefaults()
{
	//Need to remove any previously set properties.
	//Specifically any Advanced properties, since this is a new configuration
	FreeProperties(&m_cPropSets, &m_rgPropSets);
	
	//Configuration Name
	if(m_szConfigName[0] == EOL)
		strcpy(m_szConfigName, "(Default)");

	//Formulate the key
	CHAR szKeyName[MAX_NAME_LEN];
	sprintf(szKeyName, "%s\\%s", szCONFIG_KEY, m_szConfigName); 

	//Selected Provider
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Provider",			m_EnumInfo.wszName, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ParseName",			m_EnumInfo.wszParseName, MAX_NAME_LEN);

	//Provider Page
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Location",			m_wszLocation, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "DataSource",			m_wszDataSource, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "UserID",				m_wszUserID, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Password",			m_wszPassword, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "PromptProp",			(ULONG*)&m_dwPromptProp);
//	m_dwhWndProp		= (LONG)m_pCMainWindow->m_hWnd;

	//Properties Page
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ProvString",			m_wszProvString, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Catalog",			m_wszCatalog, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "lcidProp",			(ULONG*)&m_dwlcidProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "AsynchProp",			(ULONG*)&m_dwAsynchProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "TimeoutProp",		(ULONG*)&m_dwTimeoutProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ModeProp",			(ULONG*)&m_dwModeProp);

	//Security Page
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ProtectionProp",		(ULONG*)&m_dwProtectionProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ImpersonateProp",	(ULONG*)&m_dwImpersonateProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "MaskPasswordProp",	(ULONG*)&m_dwMaskPasswordProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "EncryptPasswordProp",(ULONG*)&m_dwEncryptPasswordProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "CacheProp",			(ULONG*)&m_dwCacheProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "PersistProp",		(ULONG*)&m_dwPersistProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "PersistEncryptProp",	(ULONG*)&m_dwPersistEncryptProp);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Integrated",			m_wszIntegrated, MAX_NAME_LEN);

	//Options Page
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "CLSCTX",				&m_dwCLSCTX);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "RemoteServer",		m_wszRemoteServer, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ConnectOptions",		&m_dwConnectOpts);

	//ServiceComponents
	GetRegEnumValue(HKEY_ROWSETVIEWER, szRECENTINITSTRING_KEY, 0, m_wszInitString, MAX_QUERY_LEN);
	return TRUE;
}
	

////////////////////////////////////////////////////////////////
// CConnectDlg::SaveDefaults
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::SaveDefaults()
{
	//Configuration Name
	if(m_szConfigName[0] == EOL)
		strcpy(m_szConfigName, "(Default)");

	//Save Configuration Name, so we know which configuration is the default
	SetRegEntry(HKEY_ROWSETVIEWER, szCONFIG_KEY, "DefaultConfig", m_szConfigName);

	//Formulate the key
	CHAR szKeyName[MAX_NAME_LEN];
	sprintf(szKeyName, "%s\\%s", szCONFIG_KEY, m_szConfigName); 

	//Selected Provider
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Provider",			m_EnumInfo.wszName);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ParseName",			m_EnumInfo.wszParseName);

	//Provider Page
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Location",			m_wszLocation);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "DataSource",			m_wszDataSource);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "UserID",				m_wszUserID);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Password",			m_wszPassword);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "PromptProp",			m_dwPromptProp);
//	m_dwhWndProp		= (LONG)m_pCMainWindow->m_hWnd;

	//Properties Page
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ProvString",			m_wszProvString);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Catalog",			m_wszCatalog);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "lcidProp",			m_dwlcidProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "AsynchProp",			m_dwAsynchProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "TimeoutProp",		m_dwTimeoutProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ModeProp",			m_dwModeProp);

	//Security Page
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ProtectionProp",		m_dwProtectionProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ImpersonateProp",	m_dwImpersonateProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "MaskPasswordProp",	m_dwMaskPasswordProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "EncryptPasswordProp",m_dwEncryptPasswordProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "CacheProp",			m_dwCacheProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "PersistProp",		m_dwPersistProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "PersistEncryptProp",	m_dwPersistEncryptProp);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "Integrated",			m_wszIntegrated);

	//Options Page
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "CLSCTX",				m_dwCLSCTX);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "RemoteServer",		m_wszRemoteServer);
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "ConnectOptions",		m_dwConnectOpts);

	//Other Configuration Specific Saved Information
	SetRegEntry(HKEY_ROWSETVIEWER, szKeyName, "CommandBuffer",		m_wszCmdBuffer);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::LoadRecentConfig
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::LoadRecentConfig(ULONG iRecentConfig)
{
	ASSERT(iRecentConfig < m_listConfigs.GetCount());
	
	//Set the active config
	strcpy(m_szConfigName, m_listConfigs.GetAt(m_listConfigs.FindIndex(iRecentConfig)));
	
	//Load the saved properties and Connect
	LoadDefaults();
	FullConnect();
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::AddRecentConfig
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::AddRecentConfig(CHAR* pszConfigName)
{
	//Bascially the Alogythym for updating the Recent Configurations
	//is similar to a FIFO stack.  Lastest items become #1 and all other items
	//are moved down.  The only exception is if the item already exists
	//then it is brought up to number one and all other items are reordered...
	if(pszConfigName == NULL)
		pszConfigName = m_szConfigName;

	//Make sure it doesn't already exist in the list
	RemoveRecentConfig(pszConfigName);

	//Now add it to the list
	m_listConfigs.AddHead(strDuplicate(pszConfigName));
	
	//Make sure the list is less than the Max
	if(m_listConfigs.GetCount() > MAX_RECENTCONFIGS)
	{
		pszConfigName = m_listConfigs.RemoveTail();
		SAFE_FREE(pszConfigName);
	}
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::RemoveRecentConfig
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RemoveRecentConfig(CHAR* pszConfigName)
{
	ASSERT(pszConfigName);
	
	//Make sure it doesn't allready exist in the list
	LISTPOS pos = m_listConfigs.GetHeadPosition();
	while(pos)
	{
		LISTPOS posSave = pos;
		CHAR* pszName = m_listConfigs.GetNext(pos);

		ASSERT(pszName);
		if(strcmp(pszName, pszConfigName)==0)
		{
			//Remove this item and Add to the head
			pszName = m_listConfigs.RemoveAt(posSave);
			SAFE_FREE(pszName);
			return TRUE;
		}
	}

	return TRUE;
}

	
////////////////////////////////////////////////////////////////
// CConnectDlg::LoadRecentFile
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::LoadRecentFile(ULONG iRecentFile)
{
	ASSERT(iRecentFile < m_listFiles.GetCount());
	WCHAR wszBuffer[MAX_NAME_LEN+1];

	//Obtain the Selected File
	CHAR* pszFileName = m_listFiles.GetAt(m_listFiles.FindIndex(iRecentFile));
	
	//Load the saved properties and Connect
	ConvertToWCHAR(pszFileName, wszBuffer, MAX_NAME_LEN);
	ConnectFromFile(wszBuffer);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::AddRecentFile
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::AddRecentFile(CHAR* pszFileName)
{
	//Bascially the Alogythym for updating the Recent Files
	//is similar to a FIFO stack.  Lastest items become #1 and all other items
	//are moved down.  The only exception is if the item already exists
	//then it is brought up to number one and all other items are reordered...
	ASSERT(pszFileName);

	//Make sure it doesn't already exist in the list
	RemoveRecentFile(pszFileName);

	//Now add it to the list
	m_listFiles.AddHead(strDuplicate(pszFileName));
	
	//Make sure the list is less than the Max
	if(m_listFiles.GetCount() > MAX_RECENTFILES)	
	{
		pszFileName = m_listFiles.RemoveTail();
		SAFE_FREE(pszFileName);
	}
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::RemoveRecentFile
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RemoveRecentFile(CHAR* pszFileName)
{
	ASSERT(pszFileName);
	
	//Make sure it doesn't allready exist in the list
	LISTPOS pos = m_listFiles.GetHeadPosition();
	while(pos)
	{
		LISTPOS posSave = pos;
		CHAR* pszName = m_listFiles.GetNext(pos);

		ASSERT(pszName);
		if(strcmp(pszName, pszFileName)==0)
		{
			//Remove this item and Add to the head
			pszName = m_listFiles.RemoveAt(posSave);
			SAFE_FREE(pszName);
			return TRUE;
		}
	}

	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::pIDataInitialize
//
/////////////////////////////////////////////////////////////////
IDataInitialize*	const CConnectDlg::pIDataInitialize()
{
	static BOOL fInitialized = FALSE;

	//Obtain IDataInitialize
	if(m_pIDataInitialize == NULL && !fInitialized)
	{
		CoCreateInstance(CLSID_MSDAINITIALIZE, NULL, CLSCTX_INPROC_SERVER, IID_IDataInitialize, (void**)&m_pIDataInitialize);
		fInitialized = TRUE;
	}
	
	return m_pIDataInitialize;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::pIDBPromptInitialize
//
/////////////////////////////////////////////////////////////////
IDBPromptInitialize*	const CConnectDlg::pIDBPromptInitialize()
{
	static BOOL fInitialized = FALSE;

	//Obtain IDataInitialize
	if(m_pIDBPromptInitialize == NULL && !fInitialized)
	{
		CoCreateInstance(CLSID_DataLinks, NULL, CLSCTX_INPROC_SERVER, IID_IDBPromptInitialize, (void**)&m_pIDBPromptInitialize);
		fInitialized = TRUE;
	}
	
	return m_pIDBPromptInitialize;
}



////////////////////////////////////////////////////////////////
// CConnectDlg::Display
//
/////////////////////////////////////////////////////////////////
ULONG CConnectDlg::Display(IParseDisplayName* pIParseDisplayName, ENUMINFO* pEnumInfo)
{
	//Need to first bring up connection...
	SAFE_RELEASE(m_pIUnknown)
	SAFE_RELEASE(m_pIParseDisplayName);

	//If a IParseDisplayName interface is passed in, it represents
	//the enumerator to use for instanting the object...
	//Otherwise if NULL - it uses the Root Enumerator Object...
	SAFE_ADDREF(pIParseDisplayName);
	m_pIParseDisplayName = pIParseDisplayName;

	if(pEnumInfo)
		memcpy(&m_EnumInfo, pEnumInfo, sizeof(ENUMINFO));

	//Create an image list
	HIMAGELIST hImageList = ImageList_LoadImage(m_hInst, MAKEINTRESOURCE(IDB_TOOLBAR), 16, 16, CLR_DEFAULT , IMAGE_BITMAP, LR_DEFAULTCOLOR);
	HICON hIcon = ImageList_GetIcon(hImageList, 1, ILD_NORMAL);

	//Now Display the dialog
    PROPSHEETPAGE psp[5];
    PROPSHEETHEADER psh;

    //Header
    psh.dwSize			= sizeof(PROPSHEETHEADER);
    psh.dwFlags			= PSH_USEHICON | PSH_PROPSHEETPAGE | PSH_NOAPPLYNOW;
    psh.hwndParent		= m_hWnd;
    psh.hInstance		= m_hInst;
    psh.hIcon			= hIcon;
    psh.pszCaption		= "Full Connect";
    psh.nPages			= NUMELE(psp);
    psh.nStartPage		= 0;
    psh.ppsp			= (LPCPROPSHEETPAGE) &psp;

    //Provider
	psp[0].dwSize		= sizeof(PROPSHEETPAGE);
    psp[0].dwFlags		= PSP_USETITLE;
    psp[0].hInstance	= m_hInst;
    psp[0].pszTemplate	= MAKEINTRESOURCE(IDD_FULLCONNECT_PROVIDER);
    psp[0].pszIcon		= NULL;
    psp[0].pfnDlgProc	= ProviderProc;
    psp[0].pszTitle		= "Provider";
    psp[0].lParam		= (LONG)this;

    //Properties
	psp[1].dwSize		= sizeof(PROPSHEETPAGE);
    psp[1].dwFlags		= PSP_USETITLE;
    psp[1].hInstance	= m_hInst;
    psp[1].pszTemplate	= MAKEINTRESOURCE(IDD_FULLCONNECT_PROPERTIES);
    psp[1].pszIcon		= NULL;
    psp[1].pfnDlgProc	= PropertiesProc;
    psp[1].pszTitle		= "Properties";
    psp[1].lParam		= (LONG)this;

    //Security
	psp[2].dwSize		= sizeof(PROPSHEETPAGE);
    psp[2].dwFlags		= PSP_USETITLE;
    psp[2].hInstance	= m_hInst;
    psp[2].pszTemplate	= MAKEINTRESOURCE(IDD_FULLCONNECT_SECURITY);
    psp[2].pszIcon		= NULL;
    psp[2].pfnDlgProc	= SecurityProc;
    psp[2].pszTitle		= "Security";
    psp[2].lParam		= (LONG)this;

    //Options
	psp[3].dwSize		= sizeof(PROPSHEETPAGE);
    psp[3].dwFlags		= PSP_USETITLE;
    psp[3].hInstance	= m_hInst;
    psp[3].pszTemplate	= MAKEINTRESOURCE(IDD_FULLCONNECT_OPTIONS);
    psp[3].pszIcon		= NULL;
    psp[3].pfnDlgProc	= OptionsProc;
    psp[3].pszTitle		= "Options";
    psp[3].lParam		= (LONG)this;

    //Trace
	psp[4].dwSize		= sizeof(PROPSHEETPAGE);
    psp[4].dwFlags		= PSP_USETITLE;
    psp[4].hInstance	= m_hInst;
    psp[4].pszTemplate	= MAKEINTRESOURCE(IDD_FULLCONNECT_TRACE);
    psp[4].pszIcon		= NULL;
    psp[4].pfnDlgProc	= TraceProc;
    psp[4].pszTitle		= "Trace";
    psp[4].lParam		= (LONG)this;

	//Display the Property Sheet
    PropertySheet(&psh);

	//Now display focus to the new MDIChildWindow
	SetFocus(GetWindow(m_pCMainWindow->m_hWndMDIClient, GW_CHILD));
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::ProviderProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CConnectDlg::ProviderProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE* ps = NULL;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN
			
			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			CConnectDlg* pThis = (CConnectDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitProvider(hWnd);

			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//LBN_SELCHANGE ListBox Selection change
			switch(GET_WM_COMMAND_CMD(wParam, lParam))
			{
				//Selection change in a list box occurred
				case LBN_SELCHANGE:
				{	
					//See which combo box has changed
					switch(GET_WM_COMMAND_ID(wParam, lParam))
					{
						case IDC_CONFIG:
						{
							//Obtain Config From Drop Down
							CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
							HWND hWndConfig = GetDlgItem(hWnd, IDC_CONFIG);
							
							//Obtain the Selected Combo Text
							CB_GetSelectedText(hWndConfig, pThis->m_szConfigName, MAX_NAME_LEN);
							
							//Now we need to update the FullConnect for this Config
							pThis->LoadDefaults();
							pThis->RefreshProvider();
							return 0;
						}

						case IDC_PROVIDER:
						{
							//Obtain Provider From Drop Down
							CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
							pThis->GetProviderName();

							//Reset this connection
							SAFE_RELEASE(pThis->m_pIUnknown);
							return 0;
						}
						
					}
					break;
				}
			
				return FALSE;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_DEFAULTS:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->ResetDefaults();
					pThis->RefreshOptions();
					return 0;
				}

				case IDB_CONFIG_SAVE:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					HWND hWndConfig = GetDlgItem(hWnd, IDC_CONFIG);
					
					//Obtain the Selected Combo Text
					LONG iSel = CB_GetSelectedText(hWndConfig, pThis->m_szConfigName, MAX_NAME_LEN);
					iSel = CB_SelectText(hWndConfig, pThis->m_szConfigName, TRUE);
					
					//Now update registry
					pThis->UpdateProvider();
					pThis->SaveDefaults();

					//Update the Saved Configurations (now that successfuly connected)
					pThis->AddRecentConfig();
					return 0;
				}
								
				case IDB_CONFIG_REMOVE:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					HWND hWndConfig = GetDlgItem(hWnd, IDC_CONFIG);
					
					//Obtain the Selected Combo Text
					LONG iSel = CB_GetSelectedText(hWndConfig, pThis->m_szConfigName, MAX_NAME_LEN);
					
					//Remove this item from the ComboBox
					SendMessage(hWndConfig, CB_DELETESTRING, iSel, 0);
										
					//Formulate the key
					CHAR szKeyName[MAX_NAME_LEN];
					sprintf(szKeyName, "%s\\%s", szCONFIG_KEY, pThis->m_szConfigName); 

					//Now Remove this Item from the registry
					DelRegEntry(HKEY_ROWSETVIEWER, szKeyName);
					
					//We need to make sure this is not the default Config
					szKeyName[0] = EOL;
					GetRegEntry(HKEY_ROWSETVIEWER, szCONFIG_KEY, "DefaultConfig", szKeyName, MAX_NAME_LEN);
					if(strcmp(szKeyName, pThis->m_szConfigName)==0)
						SetRegEntry(HKEY_ROWSETVIEWER, szCONFIG_KEY, "DefaultConfig", "(Default)");

					//And Remove it from our Recent Config List
					pThis->RemoveRecentConfig(pThis->m_szConfigName);
					pThis->m_szConfigName[0] = EOL;
					return 0;
				}

				case IDB_ENUM_REFRESH:
				{
					//Get the "this" pointer
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					
					//Refresh Root Rnum (Reconnect = TRUE)
					pThis->RefreshEnum(TRUE);
					return 0;
				}

				case IDB_BROWSE_LOCATION:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->m_ePropSource = DBPROP_INIT_LOCATION;
					pThis->m_hWndSource = GetDlgItem(pThis->m_hWndProvider, IDE_LOCATION);

					RECT rect;
					GetWindowRect(GetDlgItem(hWnd, IDB_BROWSE_LOCATION), &rect);

					//Display the Context Menu
					DisplayContextMenu(	pThis->m_hInst, 
								hWnd,
								IDMENU_BROWSE_OPTION, 
								rect.left,
								rect.top,
								hWnd
								);
					return 0;
				}

				case IDB_BROWSE_DATASOURCE:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->m_ePropSource = DBPROP_INIT_DATASOURCE;
					pThis->m_hWndSource = GetDlgItem(pThis->m_hWndProvider, IDE_DATASOURCE);

					RECT rect;
					GetWindowRect(GetDlgItem(hWnd, IDB_BROWSE_DATASOURCE), &rect);

					//Display the Context Menu
					DisplayContextMenu(	pThis->m_hInst, 
								hWnd,
								IDMENU_BROWSE_OPTION, 
								rect.left,
								rect.top,
								hWnd
								);
					return 0;
				}
				
				case IDMENU_BROWSE_FILE:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					CHAR* pszTitle = NULL;
					WCHAR* pwszBuffer = NULL;
					
					switch(pThis->m_ePropSource)
					{
						case DBPROP_INIT_DATASOURCE:
						{
							pszTitle = "DBPROP_INIT_DATASOURCE";
							pwszBuffer = pThis->m_wszDataSource;
							break;
						}

						case DBPROP_INIT_LOCATION:
						{
							pszTitle = "DBPROP_INIT_LOCATION";
							pwszBuffer = pThis->m_wszLocation;
							break;
						}

						default:
							ASSERT(!"Unhandled Source!");
							break;
					}

					//Display Common Dialog to obtain DataSource...
					//This is for providers that take a filename/path for this property
					if(SUCCEEDED(BrowseOpenFileName(pThis->m_hInst, hWnd, pszTitle, pwszBuffer, MAX_NAME_LEN)))
					{
						//Just update value
						CHAR szBuffer[MAX_NAME_LEN];
						ConvertToMBCS(pwszBuffer, szBuffer, MAX_NAME_LEN);
						ReplaceEditBoxSelection(pThis->m_hWndSource, szBuffer);
					}
					return 0;
				}

				case IDMENU_BROWSE_ENUM:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					DialogBoxParam(pThis->m_hInst, MAKEINTRESOURCE(IDD_BROWSE_ENUM), hWnd, BrowseEnumeratorProc, (LPARAM)pThis);
					return 0;
				}
			}
			
			return FALSE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->m_hWndProvider = hWnd;
					pThis->RefreshProvider();
					return 0;
				}

				case PSN_KILLACTIVE://Switch
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					if(!pThis->UpdateProvider())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:
				{
					Busy();
					// Make a connection using the supplied values
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->GetProviderName();
					if(FAILED(pThis->FullConnect()))
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						Busy(OFF);
						return TRUE;
					}
					
					Busy(OFF);
					return 0;
				}

				case PSN_RESET:		//CANCEL
					return 0;
			}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// CConnectDlg::PropertiesProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CConnectDlg::PropertiesProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE* ps = NULL;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN
			
			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			CConnectDlg* pThis = (CConnectDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitProperties(hWnd);

			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return 0;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_MOREPROPERTIES:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->GetAdvProperties();
					pThis->RefreshProperties();
					return 0;
				}

				case IDB_BROWSE_PROVSTRING:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->m_ePropSource = DBPROP_INIT_PROVIDERSTRING;
					pThis->m_hWndSource = GetDlgItem(pThis->m_hWndProperties, IDE_PROVSTRING);
					
					RECT rect;
					GetWindowRect(GetDlgItem(hWnd, IDB_BROWSE_PROVSTRING), &rect);

					//Display the Context Menu
					DisplayContextMenu(	pThis->m_hInst, 
								hWnd,
								IDMENU_BROWSE_OPTION, 
								rect.left,
								rect.top,
								hWnd
								);
					return 0;
				}

				case IDMENU_BROWSE_FILE:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					CHAR* pszTitle = NULL;
					WCHAR* pwszBuffer = NULL;
					
					switch(pThis->m_ePropSource)
					{
						case DBPROP_INIT_PROVIDERSTRING:
						{
							pszTitle = "DBPROP_INIT_PROVIDERSTRING";
							pwszBuffer = pThis->m_wszProvString;
							break;
						}

						default:
							ASSERT(!"Unhandled Source!");
							break;
					}

					//Display Common Dialog to obtain DataSource...
					//This is for providers that take a filename/path for this property
					if(SUCCEEDED(BrowseOpenFileName(pThis->m_hInst, hWnd, pszTitle, pwszBuffer, MAX_NAME_LEN)))
					{
						//Just update value
						CHAR szBuffer[MAX_NAME_LEN];
						ConvertToMBCS(pwszBuffer, szBuffer, MAX_NAME_LEN);
						ReplaceEditBoxSelection(pThis->m_hWndSource, szBuffer);
					}
					return 0;
				}

				case IDMENU_BROWSE_ENUM:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					DialogBoxParam(pThis->m_hInst, MAKEINTRESOURCE(IDD_BROWSE_ENUM), hWnd, BrowseEnumeratorProc, (LPARAM)pThis);
					return 0;
				}
			}
		
			return FALSE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE: //Init
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->RefreshProperties();
					return 0;
				}

				case PSN_KILLACTIVE://Switch
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					if(!pThis->UpdateProperties())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}

					return 0;
				}

				case PSN_APPLY:		//OK
					return 0;

				case PSN_RESET:		//CANCEL
					return 0;
			}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// CConnectDlg::SecurityProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CConnectDlg::SecurityProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE* ps = NULL;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN

			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			CConnectDlg* pThis = (CConnectDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitSecurity(hWnd);

			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_BROWSE_INTEGRATED:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->m_ePropSource = DBPROP_AUTH_INTEGRATED;
					pThis->m_hWndSource = GetDlgItem(pThis->m_hWndSecurity, IDE_INTEGRATED);
					
					RECT rect;
					GetWindowRect(GetDlgItem(hWnd, IDB_BROWSE_INTEGRATED), &rect);

					//Display the Context Menu
					DisplayContextMenu(	pThis->m_hInst, 
								hWnd,
								IDMENU_BROWSE_OPTION, 
								rect.left,
								rect.top,
								hWnd
								);
					return 0;
				}

				case IDMENU_BROWSE_FILE:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					CHAR* pszTitle = NULL;
					WCHAR* pwszBuffer = NULL;
					
					switch(pThis->m_ePropSource)
					{
						case DBPROP_AUTH_INTEGRATED:
						{
							pszTitle = "DBPROP_AUTH_INTEGRATED";
							pwszBuffer = pThis->m_wszIntegrated;
							break;
						}

						default:
							ASSERT(!"Unhandled Source!");
							break;
					}

					//Display Common Dialog to obtain DataSource...
					//This is for providers that take a filename/path for this property
					if(SUCCEEDED(BrowseOpenFileName(pThis->m_hInst, hWnd, pszTitle, pwszBuffer, MAX_NAME_LEN)))
					{
						//Just update value
						CHAR szBuffer[MAX_NAME_LEN];
						ConvertToMBCS(pwszBuffer, szBuffer, MAX_NAME_LEN);
						ReplaceEditBoxSelection(pThis->m_hWndSource, szBuffer);
					}
					return 0;
				}

				case IDMENU_BROWSE_ENUM:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					DialogBoxParam(pThis->m_hInst, MAKEINTRESOURCE(IDD_BROWSE_ENUM), hWnd, BrowseEnumeratorProc, (LPARAM)pThis);
					return 0;
				}
			}
		
			return FALSE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE: //Init
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->RefreshSecurity();
					return 0;
				}

				case PSN_KILLACTIVE://Switch
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					if(!pThis->UpdateSecurity())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:		//OK
					return 0;

				case PSN_RESET:		//CANCEL
					return 0;
			}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// CConnectDlg::OptionsProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CConnectDlg::OptionsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE* ps = NULL;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN
			
			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			CConnectDlg* pThis = (CConnectDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitOptions(hWnd);

			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_REMOTE_SERVER:
					EnableWindow(GetDlgItem(hWnd, IDE_REMOTESERVER), IsDlgButtonChecked(hWnd, IDB_REMOTE_SERVER));
					return 0;
			}
		
			return FALSE;
		}

		case WM_NOTIFY:
    		switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->RefreshOptions();
					return 0;
				}

				case PSN_KILLACTIVE:
				{	
					// initialize the controls
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					if(!pThis->UpdateOptions())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
				  	return 0;
				}

				case PSN_APPLY:
				  	return 0;

				case PSN_RESET:
				  	return 0;
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// CConnectDlg::TraceProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CConnectDlg::TraceProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE* ps = NULL;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN 

			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			CConnectDlg* pThis = (CConnectDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitTrace(hWnd);

			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_TEST:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					
					//Obtain Provider from DropDown
					pThis->GetProviderName();
					pThis->CreateProviderInstance();
					return 0;
				}
			}
		
			return FALSE;
		}

		case WM_NOTIFY:
    		switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->RefreshTrace();
					return 0;
				}

				case PSN_KILLACTIVE:
				{	
					//Must set the Window handle back
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					if(!pThis->UpdateTrace())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
				  	return 0;
				}

				case PSN_APPLY:
				  	return 0;

				case PSN_RESET:
				{	
					//Must set the Window handle back
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->UpdateTrace();			
					return 0;
				}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// CConnectDlg::BrowseEnumeratorProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CConnectDlg::BrowseEnumeratorProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN

			//save off the this pointer
			CConnectDlg* pThis = (CConnectDlg*)SetThis(hWnd, lParam);
			
			HWND hWndProv = GetDlgItem(hWnd, IDC_PROVIDER);
			HWND hWndListView = GetDlgItem(hWnd, IDL_LISTVIEW);
			LONG iIndex = 0;
			CEnum* pCEnum =  pThis->m_pCEnum;

			//We may need to connect to the RootEnumerator, if not done already
			if(!pCEnum->IsConnectedToRootEnum())
				pCEnum->ConnectToRootEnum();
			
			//We just need to fill in a combo box with the same values
			//from the Root Enumerator, but only displaying Enumerators...
			for(ULONG i=0; i<pCEnum->m_cEnumInfo; i++)
			{
				if(pCEnum->m_rgEnumInfo[i].wType == DBSOURCETYPE_ENUMERATOR)
				{
					//Add the name to the list
					//Since we have the CBS_SORT turned on, the order in the Combo Box does
					//not match our array, so we pass the array index (lParam) as the item data
					iIndex = wSendMessage(hWndProv, CB_ADDSTRING, 0, pCEnum->m_rgEnumInfo[i].wszName);
					SendMessage(hWndProv, CB_SETITEMDATA, (WPARAM)iIndex, (LPARAM)i);
				}
			}
		
			//Try to do a smart guess on which Enumerator.
			//Since the spec doesn't have a actual way of prgrammatically 
			//assoiciating providers with their enumerators, we will try and search
			//the enumerator name for the providers name...
			//So if using MSDASQL, it will find MSDASQL Enumerator
			iIndex = wSendMessage(hWndProv, CB_FINDSTRING, -1, pThis->m_EnumInfo.wszName);
			SendMessage(hWndProv, CB_SETCURSEL, iIndex!=CB_ERR ? iIndex : 0, 0);

			//Setup ListView
			SendMessage(hWndListView, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES);
			LV_InsertColumn(hWndListView, 0, "SOURCES_NAME");
			HIMAGELIST hImageList = ImageList_Create(16, 16, ILC_MASK, 1, 0 );
			
			//IDI_ROW_NORMAL
			HICON hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_ROW_NORMAL));
			ImageList_AddIcon(hImageList, hIcon);
			//Set image list to the ListView
			ListView_SetImageList(hWndListView, hImageList, LVSIL_SMALL);

			//AutoSize Header
			SendMessage(hWndListView, LVM_SETCOLUMNWIDTH, 0, (LPARAM)LVSCW_AUTOSIZE_USEHEADER);

			//Send a selection change to the ComboBox so it updates everything
			SendMessage(hWnd, WM_COMMAND, GET_WM_COMMAND_MPS(IDC_PROVIDER, hWnd, LBN_SELCHANGE));
			CenterDialog(hWnd);

			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//LBN_SELCHANGE ListBox Selection change
			switch(GET_WM_COMMAND_CMD(wParam, lParam))
			{
				//Selection change in a list box occurred
				case LBN_SELCHANGE:
				{	
					//Get the "this" pointer
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					CEnum* pCEnum =  pThis->m_pCEnum;

					HWND hWndProv = GetDlgItem(hWnd, IDC_PROVIDER);
					HWND hWndListView = GetDlgItem(hWnd, IDL_LISTVIEW);
					
					//The Selection has changed...
					SendMessage(hWndListView, LVM_DELETEALLITEMS, 0, 0);
					EnableWindow(hWndListView, FALSE);
					
					//Update the Window Title - with the Description
					LONG iIndex = SendMessage(hWndProv, CB_GETCURSEL, 0, 0);
					LONG lParam = SendMessage(hWndProv, CB_GETITEMDATA, iIndex, 0);

					if(lParam!=CB_ERR && lParam <(LONG)pCEnum->m_cEnumInfo)
					{
						wSendMessage(hWnd, WM_SETTEXT, 0, pCEnum->m_rgEnumInfo[lParam].wszDescription);
					}
					return 0;
				}
			}
			
			switch (GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_CONNECT:
				{
					//Get the "this" pointer
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);

					//Fill in Enum Dialog
					pThis->DisplayBrowseEnumInfo(hWnd);
					return 0;
				}
								
				case IDOK:
				{
					//Get the "this" pointer
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					CHAR szBuffer[MAX_NAME_LEN];
					szBuffer[0] = EOL;

					//Need to find the Selected Item...
					HWND hWndListView = GetDlgItem(hWnd, IDL_LISTVIEW);
					LONG iSelRow = SendMessage(hWndListView, LVM_GETNEXTITEM, (WPARAM)-1, (LPARAM)LVNI_SELECTED);
					
					//Need to obtain Enum selection from the ListView
					LV_GetItemText(hWndListView, iSelRow, 0, szBuffer, MAX_NAME_LEN);

					//Now need to copy value to the EditBox
					ReplaceEditBoxSelection(pThis->m_hWndSource, szBuffer);
					EndDialog(hWnd, TRUE);
					return 0;
				}

				case IDCANCEL:
				{
					EndDialog(hWnd, FALSE);
					return 0;
				}
			}
			break;
		}//WM_COMMAND

	
		case WM_NOTIFY:
		{
			LV_DISPINFO* pDispInfo = (LV_DISPINFO*)lParam;
			NM_LISTVIEW* pListView = (NM_LISTVIEW*)lParam;
			
			switch(pDispInfo->hdr.code)
			{
				//Since we have "TwoClickActive" on this will get sent
				//Whenever a row is clicked on twice!
				//This functionality used to be done with NM_DBCLK
				case LVN_ITEMACTIVATE:
				{
					//Send an OK message
					SendMessage(hWnd, WM_COMMAND, GET_WM_COMMAND_MPS(IDOK, hWnd, 0));
					return 0;
				}
			}
			break;
		}//WM_NOTIFY
	}

	return FALSE;   
}



////////////////////////////////////////////////////////////////
// CConnectDlg::GetDataSourceProc
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::GetDataSourceProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static LONG iSavedSel = 2;		//IDBInitialize

	switch(msg)
	{
		case WM_INITDIALOG:
		{
			Busy();
			EXC_BEGIN

			//Save the "this" pointer
			CConnectDlg* pThis = (CConnectDlg*)SetThis(hWnd, (LPARAM)lParam);
			HWND hWndInitString = GetDlgItem(hWnd, IDC_INITSTRING);
			HWND hWndInterface	= GetDlgItem(hWnd, IDC_INTERFACE);
			LONG iSel = 0;

			//Set the Saved InitString
			wSendMessage(hWndInitString, WM_SETTEXT, 0, pThis->m_wszInitString);

			//Interface List...
			for(ULONG i=0; i<g_cInterfaceMaps; i++)
			{
				//Interface Name
				iSel = SendMessage(hWndInterface, CB_ADDSTRING,	0, (LPARAM)g_rgInterfaceMap[i].pszName);
				SendMessage(hWndInterface, CB_SETITEMDATA, iSel, (LPARAM)&g_rgInterfaceMap[i]);
			}

			//Set Default...
			SendMessage(hWndInterface, CB_SETCURSEL, iSavedSel, 0);
			CenterDialog(hWnd);
			Busy(OFF);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			switch (GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_OPTIONS:
				{
					//Get the "this" pointer
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);

					//Bring up the Connection Options
					pThis->GetOptionsObj()->Display();
					return 0;
				}

				case IDOK:
				{
					//Get the "this" pointer
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);

					HWND hWndInitString = GetDlgItem(hWnd, IDC_INITSTRING);
					HWND hWndInterface	= GetDlgItem(hWnd, IDC_INTERFACE);
					HRESULT hr = S_OK;

					//Get the InitString from the EditBox
					wSendMessage(hWndInitString, WM_GETTEXT, MAX_QUERY_LEN, pThis->m_wszInitString);
					
					//Interface
					iSavedSel = SendMessage(hWndInterface, CB_GETCURSEL, 0, 0);
					GUIDMAP* pGuidMap = (GUIDMAP*)SendMessage(hWndInterface, CB_GETITEMDATA, iSavedSel, 0);

					//GetDataSource - with the specified InitString
					TESTC(hr = pThis->ConnectFromString(pThis->m_wszInitString, NULL, *(pGuidMap->pGuid)));
					
				CLEANUP:
					SetFocus(hWnd);
					if(SUCCEEDED(hr))
						EndDialog(hWnd, TRUE);
					return 0;
				}

				case IDB_BROWSE_PROVIDER:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					pThis->m_hWndSource = GetDlgItem(hWnd, IDC_INITSTRING);
					
					RECT rect;
					GetWindowRect(GetDlgItem(hWnd, IDB_BROWSE_PROVIDER), &rect);

					//Display the Context Menu
					DisplayContextMenu(	pThis->m_hInst, 
								hWnd,
								IDMENU_BROWSE_OPTION, 
								rect.left,
								rect.top,
								hWnd
								);
					return 0;
				}

				case IDMENU_BROWSE_FILE:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					WCHAR wszFileName[MAX_NAME_LEN];

					//Display Common Dialog to obtain DataSource...
					//This is for providers that take a filename/path for this property
					if(SUCCEEDED(BrowseOpenFileName(pThis->m_hInst, hWnd, "Browse for File", wszFileName, MAX_NAME_LEN)))
					{
						//Just update value
						CHAR szBuffer[MAX_NAME_LEN];
						ConvertToMBCS(wszFileName, szBuffer, MAX_NAME_LEN);
						ReplaceEditBoxSelection(pThis->m_hWndSource, szBuffer);
					}
					return 0;
				}

				case IDMENU_BROWSE_ENUM:
				{
					CConnectDlg* pThis = (CConnectDlg*)GetThis(hWnd);
					DialogBoxParam(pThis->m_hInst, MAKEINTRESOURCE(IDD_BROWSE_ENUM), hWnd, BrowseEnumeratorProc, (LPARAM)pThis);
					return 0;
				}

				case IDCANCEL:
				{
					EndDialog(hWnd, FALSE);
					return 0;
				}
			}
			break;
		}//WM_COMMAND
	}

	return FALSE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::DisplayBrowseEnumInfo
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::DisplayBrowseEnumInfo(HWND hWnd)
{
	ASSERT(m_pCEnum);
	HRESULT hr = S_OK;
	
	ULONG i,cEnumInfo = 0;
	ENUMINFO* rgEnumInfo = NULL;

	CHAR szBuffer[MAX_NAME_LEN];
	WCHAR wszBuffer[MAX_NAME_LEN];
	IParseDisplayName* pIParseDisplayName = NULL;
	WCHAR* pwszParseName = NULL;

	HWND hWndProv = GetDlgItem(hWnd, IDC_PROVIDER);
	HWND hWndListView = GetDlgItem(hWnd, IDL_LISTVIEW);

	//Clear Previous Items
	SendMessage(hWndListView, LVM_DELETEALLITEMS, 0, 0);
	EnableWindow(hWndListView, FALSE);

	//Obtain the Specified ParseName
	LONG iSel = SendMessage(hWndProv, CB_GETCURSEL, 0, 0);
	if(iSel == CB_ERR)
	{
		wszBuffer[0] = wEOL;
		wSendMessage(hWndProv, WM_GETTEXT, MAX_NAME_LEN, wszBuffer);
		pwszParseName = wszBuffer;
	}
	else
	{
		LONG iEnumIndex = SendMessage(hWndProv, CB_GETITEMDATA, iSel, 0);
		pwszParseName = m_pCEnum->GetEnumInfo(iEnumIndex)->wszParseName;
	}


	//Connect to the specified Enumerator
	TESTC(hr = m_pCEnum->CreateProvider(NULL, pwszParseName, CLSCTX_INPROC_SERVER, IID_IParseDisplayName, (IUnknown**)&pIParseDisplayName, 0));

	//Obtain the Enumerator Rowset
	//Some providers may have a problem with all rows, but we may have
	//Actually retrived some rows, so just display as many as we got...
	hr = m_pCEnum->GetEnumRowset(pIParseDisplayName, &cEnumInfo, &rgEnumInfo);
	if(FAILED(hr) && cEnumInfo==0)
		goto CLEANUP;

	//Now loop through the EnumInfo and Display in the ListView
	for(i=0; i<cEnumInfo; i++)
	{
		//SOURCES_NAME
		ConvertToMBCS(rgEnumInfo[i].wszName, szBuffer, MAX_NAME_LEN);
		LV_InsertItem(hWndListView, i, 0, szBuffer, 0, 0);
	}
	
	EnableWindow(hWndListView, TRUE);

CLEANUP:
	SAFE_FREE(rgEnumInfo);
	SAFE_RELEASE(pIParseDisplayName);
	return hr;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::InitPropCombo
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::InitPropCombo(HWND hWndCombo, ULONG cItems, NAMEMAP* rgNameMap, LONG lParam)
{
	//Remove any existing Data
	SendMessage(hWndCombo,	CB_RESETCONTENT, 0, 0);
	
	//First Item of every Property Combo is "No Set"
	LONG iSel = SendMessage(hWndCombo, CB_ADDSTRING, 0, (LPARAM)"Not Set");
	SendMessage(hWndCombo,	CB_SETITEMDATA,	iSel, (LPARAM)NOTSET);
	
	//Fill in all Combo values...
	for(ULONG i=0; i<cItems; i++)
	{
		iSel = SendMessage(hWndCombo, CB_ADDSTRING, 0, (LPARAM)rgNameMap[i].pszName);
		SendMessage(hWndCombo, CB_SETITEMDATA,	iSel, (LPARAM)rgNameMap[i].lItem);
	}
	
	//Set the specified Default
	CB_SelectItemValue(hWndCombo, lParam);
	return S_OK;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::InitProvider
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::InitProvider(HWND hWnd)
{
	m_hWndProvider  = hWnd;
	m_hWndProperties= NULL;
	m_hWndSecurity	= NULL;
	m_hWndOptions	= NULL;
	m_hWndTrace		= NULL;
	
	HWND hWndConfig	= GetDlgItem(hWnd, IDC_CONFIG);
	HWND hWndProv	= GetDlgItem(hWnd, IDC_PROVIDER);
	HWND hWndPrompt	= GetDlgItem(hWnd, IDC_PROMPT);
	CHAR szBuffer[MAX_NAME_LEN];

	//Setup Button Icons
//	HBITMAP hBitmap = LoadBitmap(m_hInst, MAKEINTRESOURCE(IDB_READONLY));
//	SendMessage(GetDlgItem(hWnd, IDB_BROWSE_LOCATION), BM_SETIMAGE, (WPARAM)IMAGE_BITMAP, (LPARAM)hBitmap);
//	SendMessage(GetDlgItem(hWnd, IDB_BROWSE_DATASOURCE), BM_SETIMAGE, (WPARAM)IMAGE_BITMAP, (LPARAM)hBitmap);
			
	//Reset ListBox information
	m_pCListBox->ClearNotifications();

	//Enable Combo if using RootEnum
	EnableWindow(hWndConfig,	(BOOL)m_pIParseDisplayName==NULL);
	EnableWindow(hWndProv,		(BOOL)m_pIParseDisplayName==NULL);

	//Enable Save/Remove Buttons only if at the "base" level, using RootEnum or 
	//CoCreate, we can't enable these buttons for instaces DataSource or this is
	//a drill down from an enumerator, since we can't get back to this state without
	//the Enumerator instace...
	EnableWindow(GetDlgItem(hWnd, IDB_CONFIG_SAVE),		(BOOL)m_pIParseDisplayName==NULL);
	EnableWindow(GetDlgItem(hWnd, IDB_CONFIG_REMOVE),	(BOOL)m_pIParseDisplayName==NULL);
	EnableWindow(GetDlgItem(hWnd, IDB_ENUM_REFRESH),	(BOOL)m_pIParseDisplayName==NULL);

	//Fill in Enumerator list
	RefreshEnum();

	//DBPROP_INIT_PROMPT
	const static NAMEMAP rgPrompts[] =
	{
		VALUE_CHAR(DBPROMPT_PROMPT),
		VALUE_CHAR(DBPROMPT_COMPLETE),
		VALUE_CHAR(DBPROMPT_COMPLETEREQUIRED),
		VALUE_CHAR(DBPROMPT_NOPROMPT),
	};

	//Fill in Prompt Combo
	InitPropCombo(hWndPrompt, NUMELE(rgPrompts), (NAMEMAP*)rgPrompts, m_dwPromptProp == NOTSET ? 0 : m_dwPromptProp);

	//Load all items from the Registry
	//Limit it to the first 200 registry entries!
	//So we don't spend all day loading this combo box!
	for(ULONG i=0; i<200; i++)
	{
		if(FAILED(GetRegEnumKey(HKEY_ROWSETVIEWER, szCONFIG_KEY, i, szBuffer, MAX_NAME_LEN)))
			break;
		
		SendMessage(hWndConfig, CB_ADDSTRING, 0, (LPARAM)szBuffer);
	}

	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::RefreshEnum
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RefreshEnum(BOOL fReconnect)
{
	HWND hWndProv	= GetDlgItem(m_hWndProvider, IDC_PROVIDER);
	LONG iIndex = 0;

	//Remove all previous entries
	SendMessage(hWndProv, CB_RESETCONTENT, 0, 0);

	//Initialize the Enumerator to establish a list of 
	//Providers and Enumerators, if we don'thave them already
	if(m_pIParseDisplayName == NULL)
	{
		//No need to check the HRESULT since CEnum is capable of instatiing
		//objects, even without the RootEnum...
		if(fReconnect || !m_pCEnum->IsConnectedToRootEnum())
			m_pCEnum->ConnectToRootEnum();

		//Fill out the provider name combo box.
		for(ULONG i=0; i<m_pCEnum->m_cEnumInfo; i++)
		{
			//Add the name to the list
			//Since we have the CBS_SORT turned on, the order in the Combo Box does
			//not match our array, so we pass the array index (lParam) as the item data
			iIndex = wSendMessage(hWndProv, CB_ADDSTRING, 0, m_pCEnum->m_rgEnumInfo[i].wszName);
			SendMessage(hWndProv, CB_SETITEMDATA, (WPARAM)iIndex, (LPARAM)i);
		}
	
		//Try to find Previous Selection by default
		CB_SelectText(hWndProv, m_EnumInfo.wszName, FALSE);
	}
	else
	{
		wSendMessage(hWndProv, WM_SETTEXT, 0, m_EnumInfo.wszName);
	}

	//Obtain Provider
	GetProviderName();
	return TRUE;
}

	
////////////////////////////////////////////////////////////////
// CConnectDlg::RefreshProvider
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RefreshProvider()
{
	HWND hWnd		= m_hWndProvider;
	HWND hWndConfig	= GetDlgItem(hWnd, IDC_CONFIG);
	HWND hWndProv	= GetDlgItem(hWnd, IDC_PROVIDER);
	HWND hWndPrompt	= GetDlgItem(hWnd, IDC_PROMPT);
	LONG iIndex = 0;

	//Restore Saved Provider
	iIndex = CB_SelectText(hWndProv, m_EnumInfo.wszName, FALSE);

	//Obtain Provider from DropDown
	GetProviderName();

	//Restore Saved Values - Properties
	wSendMessage(GetDlgItem(hWnd, IDE_LOCATION),	WM_SETTEXT, 0, m_wszLocation);
	wSendMessage(GetDlgItem(hWnd, IDE_DATASOURCE),	WM_SETTEXT, 0, m_wszDataSource);
	wSendMessage(GetDlgItem(hWnd, IDE_USERID),		WM_SETTEXT, 0, m_wszUserID);
	wSendMessage(GetDlgItem(hWnd, IDE_PASSWORD),	WM_SETTEXT, 0, m_wszPassword);

	//Set Prompt Combo
	CB_SelectItemValue(hWndPrompt, m_dwPromptProp);

	//Try to find our Defaut Configuration
	iIndex = CB_SelectText(hWndConfig, m_szConfigName, TRUE);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::UpdateProvider
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::UpdateProvider()
{
	//Retrieve "Common" DBINIT Propertiess from the Provider Sheet
	HWND hWnd = m_hWndProvider;
	HWND hWndConfig = GetDlgItem(hWnd, IDC_CONFIG);
	HWND hWndPrompt = GetDlgItem(hWnd, IDC_PROMPT);

	//Obtain Provider Selection
	GetProviderName();

	//Obtain Property Values
	wSendMessage(GetDlgItem(hWnd, IDE_LOCATION),	WM_GETTEXT, MAX_NAME_LEN, m_wszLocation);
	wSendMessage(GetDlgItem(hWnd, IDE_DATASOURCE),WM_GETTEXT, MAX_NAME_LEN, m_wszDataSource);
	wSendMessage(GetDlgItem(hWnd, IDE_USERID),	WM_GETTEXT, MAX_NAME_LEN, m_wszUserID);
	wSendMessage(GetDlgItem(hWnd, IDE_PASSWORD),	WM_GETTEXT, MAX_NAME_LEN, m_wszPassword);
	
	//Obtain Prompt Value
	LONG iSel = SendMessage(hWndPrompt, CB_GETCURSEL, 0, 0);
	m_dwPromptProp = SendMessage(hWndPrompt, CB_GETITEMDATA, iSel, 0);

	//Obtain Configuration Name
	CB_GetSelectedText(hWndConfig, m_szConfigName, MAX_NAME_LEN);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::InitProperties
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::InitProperties(HWND hWnd)
{
	m_hWndProperties	= hWnd;
	HWND hWndAsynch		= GetDlgItem(hWnd, IDC_ASYNCH);
	HWND hWndLocale		= GetDlgItem(hWnd, IDC_LOCALE);
	HWND hWndMode		= GetDlgItem(hWnd, IDL_MODE);

	//DBPROP_INIT_ASYNCH
	const static NAMEMAP rgAsynch[] =
	{
		VALUE_CHAR(DBPROPVAL_ASYNCH_INITIALIZE),
	};

	//Fill in AsynchCombo
	InitPropCombo(hWndAsynch, NUMELE(rgAsynch), (NAMEMAP*)rgAsynch, m_dwAsynchProp);

	//These values are lined up inorder of values.
	//DBPROP_INIT_LOCALE = 0x1...
	const static NAMEMAP rgLocale[] =
	{
		VALUE_CHAR(LOCALE_SYSTEM_DEFAULT),
		VALUE_CHAR(LOCALE_USER_DEFAULT),
		VALUE_CHAR(LOCALE_NEUTRAL),
		VALUE_CHAR(GetSystemDefaultLCID()),
		VALUE_CHAR(GetUserDefaultLCID()),
	};

	//Fill in LocaleCombo
	InitPropCombo(hWndLocale, NUMELE(rgLocale), (NAMEMAP*)rgLocale, m_dwlcidProp);

	//DBPROP_INIT_MODE
	const static NAMEMAP rgMode[] =
	{
		VALUE_CHAR(DB_MODE_READ),
		VALUE_CHAR(DB_MODE_WRITE),
		VALUE_CHAR(DB_MODE_READWRITE),
		VALUE_CHAR(DB_MODE_SHARE_DENY_READ),
		VALUE_CHAR(DB_MODE_SHARE_DENY_WRITE),
		VALUE_CHAR(DB_MODE_SHARE_EXCLUSIVE),
		VALUE_CHAR(DB_MODE_SHARE_DENY_NONE),
	};

	//Fill in Mode ListBox
	SendMessage(hWndMode, LB_RESETCONTENT, 0, 0);
	for(ULONG i=0; i<NUMELE(rgMode); i++)
	{
		LONG iSel = SendMessage(hWndMode, LB_ADDSTRING, 0, (LPARAM)rgMode[i].pszName);
		SendMessage(hWndMode,	LB_SETITEMDATA,	iSel, (LPARAM)rgMode[i].lItem);
	}

	//Restore Saved Values
	return RefreshProperties();
}


////////////////////////////////////////////////////////////////
// CConnectDlg::RefreshProperties
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RefreshProperties()
{
	HWND hWnd			= m_hWndProperties;
	HWND hWndAsynch		= GetDlgItem(hWnd, IDC_ASYNCH);
	HWND hWndLocale		= GetDlgItem(hWnd, IDC_LOCALE);
	HWND hWndMode		= GetDlgItem(hWnd, IDL_MODE);

	//Restore Saved Values - Properties
	wSendMessage(GetDlgItem(hWnd, IDE_PROVSTRING),	WM_SETTEXT, 0, m_wszProvString);
	wSendMessage(GetDlgItem(hWnd, IDC_CATALOG),		WM_SETTEXT, 0, m_wszCatalog);
	
	//Timeout
	if(m_dwTimeoutProp != NOTSET)
		wSendMessageFmt(GetDlgItem(hWnd, IDE_TIMEOUT),	WM_SETTEXT, 0, L"%d", m_dwTimeoutProp);

	//Fill in AsynchCombo
	CB_SelectItemValue(hWndAsynch, m_dwAsynchProp);

	//Fill in LocaleCombo
	CB_SelectItemValue(hWndLocale, m_dwlcidProp);

	//Fill in Mode ListBox
	LONG iCount = SendMessage(hWndMode, LB_GETCOUNT, 0, 0);
	for(LONG i=0; i<iCount; i++)
	{
		//Reselect Mode Options (turn it on or off...)
		LONG lParam = SendMessage(hWndMode,	LB_GETITEMDATA,	i, 0);
		
		if(m_dwModeProp == NOTSET)
			 SendMessage(hWndMode, LB_SETSEL, 0, i); //Turn off
		else
			 SendMessage(hWndMode, LB_SETSEL, (lParam & m_dwModeProp) == lParam, i);
	}

	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::UpdateProperties
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::UpdateProperties()
{
	//Retrieve Values from Properties Sheet
	HWND hWnd = m_hWndProperties;
	HWND hWndAsynch = GetDlgItem(hWnd, IDC_ASYNCH);
	HWND hWndLocale = GetDlgItem(hWnd, IDC_LOCALE);
	HWND hWndMode	= GetDlgItem(hWnd, IDL_MODE);

	wSendMessage(GetDlgItem(hWnd, IDE_PROVSTRING),	WM_GETTEXT, MAX_NAME_LEN, m_wszProvString);
	wSendMessage(GetDlgItem(hWnd, IDC_CATALOG),		WM_GETTEXT, MAX_NAME_LEN, m_wszCatalog);
	
	//Obtain Asynch Value
	LONG iSel = SendMessage(hWndAsynch, CB_GETCURSEL, 0, 0);
	m_dwAsynchProp = SendMessage(hWndAsynch, CB_GETITEMDATA, iSel, 0);

	//Obtain Locale Value
	iSel = SendMessage(hWndLocale, CB_GETCURSEL, 0, 0);
	m_dwlcidProp = SendMessage(hWndLocale, CB_GETITEMDATA, iSel, 0);

	//Timeout
	m_dwTimeoutProp = NOTSET;
	if(!GetEditBoxValue(GetDlgItem(hWnd, IDE_TIMEOUT), 0, LONG_MAX-1, &m_dwTimeoutProp, TRUE))
	{
		m_dwTimeoutProp = NOTSET;
		return FALSE;
	}
	
	//Obtain all Mode Selected Items...
	LONG iSelCount = SendMessage(hWndMode, LB_GETSELCOUNT, 0, 0);
	ASSERT(iSelCount < 20);
	LONG rgSelItems[20];
	SendMessage(hWndMode, LB_GETSELITEMS, (WPARAM)20, (LPARAM)rgSelItems);

	m_dwModeProp = iSelCount ? 0 : NOTSET;
	for(LONG i=0; i<iSelCount; i++)
	{
		m_dwModeProp |= SendMessage(hWndMode, LB_GETITEMDATA, rgSelItems[i], 0);
	}

	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::GetAdvProperties
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::GetAdvProperties()
{
	HRESULT hr = S_OK;
	IDBProperties* pIDBProperties = NULL;
	
	ULONG i,j = 0;

	//Obtain instance of Provider/Enum (if we haven't done so already)
	TESTC(hr = GetProviderName());
	TESTC(hr = CreateProviderInstance());

	//GetProperties (incase we haven't already)
	TESTC(hr = GetPropSets());
	
	//Bring up the SetProperties Dialog
	ASSERT(m_pIUnknown);
	TESTC(hr = m_pIUnknown->QueryInterface(IID_IDBProperties, (void**)&pIDBProperties));
	TESTC(hr = m_pCPropDlg->SetProperties(m_hWnd, DATASOURCEINIT, pIDBProperties, pIDBProperties, &m_cPropSets, &m_rgPropSets));

	//Now readjust our saved values...
	//Add update Dialogs, we only need to update values our dialogs know
	//about, all other properties are just saved in the set and added later...
	for(i=0; i<m_cPropSets; i++)
	{
		for(j=0; j<m_rgPropSets[i].cProperties; j++)
		{
			DBPROP* pProp = &m_rgPropSets[i].rgProperties[j];
			switch(pProp->dwPropertyID)
			{
				//Provider Page
				case DBPROP_INIT_LOCATION:
					VariantToString(&pProp->vValue, m_wszLocation, MAX_NAME_LEN, CONV_VARBOOL);
					break;

				case DBPROP_INIT_DATASOURCE:
					VariantToString(&pProp->vValue, m_wszDataSource, MAX_NAME_LEN, CONV_VARBOOL);
					break;

				case DBPROP_AUTH_USERID:
					VariantToString(&pProp->vValue, m_wszUserID, MAX_NAME_LEN, CONV_VARBOOL);
					break;

				case DBPROP_AUTH_PASSWORD:
					VariantToString(&pProp->vValue, m_wszPassword, MAX_NAME_LEN, CONV_VARBOOL);
					break;

				case DBPROP_INIT_PROMPT:
					m_dwPromptProp = V_I2(&pProp->vValue);
					break;

				case DBPROP_INIT_HWND:
					m_dwhWndProp = V_I4(&pProp->vValue);
					break;
				
				//Propeerties Page
				case DBPROP_INIT_PROVIDERSTRING:
					VariantToString(&pProp->vValue, m_wszProvString, MAX_NAME_LEN, CONV_VARBOOL);
					break;

				case DBPROP_INIT_CATALOG:
					VariantToString(&pProp->vValue, m_wszCatalog, MAX_NAME_LEN, CONV_VARBOOL);
					break;

				case DBPROP_INIT_LCID:
					m_dwlcidProp = V_I4(&pProp->vValue);
					break;
				
				case DBPROP_INIT_ASYNCH:
					m_dwAsynchProp = V_I4(&pProp->vValue);
					break;
				
				case DBPROP_INIT_TIMEOUT:
					m_dwTimeoutProp = V_I4(&pProp->vValue);
					break;
		
				case DBPROP_INIT_MODE:
					m_dwModeProp = V_I4(&pProp->vValue);
					break;
				
				//Security Page
				case DBPROP_INIT_PROTECTION_LEVEL:
					m_dwProtectionProp = V_I4(&pProp->vValue);
					break;

				case DBPROP_INIT_IMPERSONATION_LEVEL:
					m_dwImpersonateProp = V_I4(&pProp->vValue);
					break;

				case DBPROP_AUTH_MASK_PASSWORD:
					m_dwMaskPasswordProp = V_BOOL(&pProp->vValue);
					break;

				case DBPROP_AUTH_ENCRYPT_PASSWORD:
					m_dwEncryptPasswordProp = V_BOOL(&pProp->vValue);
					break;

				case DBPROP_AUTH_CACHE_AUTHINFO:
					m_dwCacheProp = V_BOOL(&pProp->vValue);
					break;

				case DBPROP_AUTH_PERSIST_SENSITIVE_AUTHINFO:
					m_dwPersistProp = V_BOOL(&pProp->vValue);
					break;

				case DBPROP_AUTH_PERSIST_ENCRYPTED:
					m_dwPersistEncryptProp = V_BOOL(&pProp->vValue);
					break;

				case DBPROP_AUTH_INTEGRATED:
					VariantToString(&pProp->vValue, m_wszIntegrated, MAX_NAME_LEN, CONV_VARBOOL);
					break;
			}
		}
	};

CLEANUP:
	SAFE_RELEASE(pIDBProperties);
	return hr;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::InitSecurity
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::InitSecurity(HWND hWnd)
{
	m_hWndSecurity	= hWnd;
	HWND hWndProtection = GetDlgItem(hWnd, IDC_PROTECTION);
	HWND hWndImpersonation = GetDlgItem(hWnd, IDC_IMPERSONATE);

	//DBPROP_INIT_PROTECTION_LEVEL
	const static NAMEMAP rgProtection[] =
	{
		VALUE_CHAR(DB_PROT_LEVEL_NONE),
		VALUE_CHAR(DB_PROT_LEVEL_CONNECT),
		VALUE_CHAR(DB_PROT_LEVEL_CALL),
		VALUE_CHAR(DB_PROT_LEVEL_PKT),
		VALUE_CHAR(DB_PROT_LEVEL_PKT_INTEGRITY),
		VALUE_CHAR(DB_PROT_LEVEL_PKT_PRIVACY),
	};

	//Fill in ProtectionCombo
	InitPropCombo(hWndProtection, NUMELE(rgProtection), (NAMEMAP*)rgProtection, m_dwProtectionProp);

	//DBPROP_INIT_IMPERSONATION_LEVEL
	const static NAMEMAP rgImpersonation[] =
	{
		VALUE_CHAR(DB_IMP_LEVEL_ANONYMOUS),
		VALUE_CHAR(DB_IMP_LEVEL_IDENTIFY),
		VALUE_CHAR(DB_IMP_LEVEL_IMPERSONATE),
		VALUE_CHAR(DB_IMP_LEVEL_DELEGATE),
	};

	//Fill in ImpersonationCombo
	InitPropCombo(hWndImpersonation, NUMELE(rgImpersonation), (NAMEMAP*)rgImpersonation, m_dwImpersonateProp);

	//Restore Saved Values
	return RefreshSecurity();
}


////////////////////////////////////////////////////////////////
// CConnectDlg::RefreshSecurity
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RefreshSecurity()
{
	HWND hWnd				= m_hWndSecurity;
	HWND hWndProtection		= GetDlgItem(hWnd, IDC_PROTECTION);
	HWND hWndImpersonation	= GetDlgItem(hWnd, IDC_IMPERSONATE);

	//Fill in ProtectionCombo
	CB_SelectItemValue(hWndProtection, m_dwProtectionProp);

	//Fill in ImpersonationCombo
	CB_SelectItemValue(hWndImpersonation, m_dwImpersonateProp);

	//Restore Saved Values
	CheckDlgButton(hWnd, IDB_MASKPASSWORD,		m_dwMaskPasswordProp == VARIANT_TRUE ? BST_CHECKED : m_dwMaskPasswordProp == VARIANT_FALSE ? BST_INDETERMINATE : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_ENCRYPTPASSWORD,	m_dwEncryptPasswordProp == VARIANT_TRUE ? BST_CHECKED : m_dwEncryptPasswordProp == VARIANT_FALSE ? BST_INDETERMINATE : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_CACHEINFO,			m_dwCacheProp == VARIANT_TRUE ? BST_CHECKED : m_dwCacheProp == VARIANT_FALSE ? BST_INDETERMINATE : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_PERSISTINFO,		m_dwPersistProp == VARIANT_TRUE ? BST_CHECKED : m_dwPersistProp == VARIANT_FALSE ? BST_INDETERMINATE : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_ENCRYPTINFO,		m_dwPersistEncryptProp == VARIANT_TRUE ? BST_CHECKED : m_dwPersistEncryptProp == VARIANT_FALSE ? BST_INDETERMINATE : BST_UNCHECKED);
	
	//Restore Save Values - Integrated
	wSendMessage(GetDlgItem(hWnd, IDE_INTEGRATED),	WM_SETTEXT, 0, m_wszIntegrated);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::UpdateSecurity
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::UpdateSecurity()
{
	//Retrieve Values from Security Sheet
	HWND hWnd = m_hWndSecurity;
	HWND hWndProtection = GetDlgItem(hWnd, IDC_PROTECTION);
	HWND hWndImpersonate = GetDlgItem(hWnd, IDC_IMPERSONATE);

	//Obtain Protection Value
	LONG iSel = SendMessage(hWndProtection, CB_GETCURSEL, 0, 0);
	m_dwProtectionProp = SendMessage(hWndProtection, CB_GETITEMDATA, iSel, 0);

	//Obtain Impersonate Value
	iSel = SendMessage(hWndImpersonate, CB_GETCURSEL, 0, 0);
	m_dwImpersonateProp = SendMessage(hWndImpersonate, CB_GETITEMDATA, iSel, 0);

	//MaskPassword
	m_dwMaskPasswordProp = IsDlgButtonChecked(hWnd, IDB_MASKPASSWORD) == BST_CHECKED ? VARIANT_TRUE : IsDlgButtonChecked(hWnd, IDB_MASKPASSWORD) == BST_INDETERMINATE ? VARIANT_FALSE : NOTSET;

	//m_dwEncryptPasswordProp
	m_dwEncryptPasswordProp  = IsDlgButtonChecked(hWnd, IDB_ENCRYPTPASSWORD) == BST_CHECKED ? VARIANT_TRUE : IsDlgButtonChecked(hWnd, IDB_ENCRYPTPASSWORD) == BST_INDETERMINATE ? VARIANT_FALSE : NOTSET;

	//m_dwCacheProp
	m_dwCacheProp = IsDlgButtonChecked(hWnd, IDB_CACHEINFO) == BST_CHECKED ? VARIANT_TRUE : IsDlgButtonChecked(hWnd, IDB_CACHEINFO) == BST_INDETERMINATE ? VARIANT_FALSE : NOTSET;

	//m_dwPersistProp
	m_dwPersistProp = IsDlgButtonChecked(hWnd, IDB_PERSISTINFO) == BST_CHECKED ? VARIANT_TRUE : IsDlgButtonChecked(hWnd, IDB_PERSISTINFO) == BST_INDETERMINATE ? VARIANT_FALSE : NOTSET;

	//m_dwPersistEncryptProp
	m_dwPersistEncryptProp = IsDlgButtonChecked(hWnd, IDB_ENCRYPTINFO) == BST_CHECKED ? VARIANT_TRUE : IsDlgButtonChecked(hWnd, IDB_ENCRYPTINFO) == BST_INDETERMINATE ? VARIANT_FALSE : NOTSET;

	wSendMessage(GetDlgItem(hWnd, IDE_INTEGRATED),	WM_GETTEXT, MAX_NAME_LEN, m_wszIntegrated);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::InitOptions
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::InitOptions(HWND hWnd)
{
	m_hWndOptions = hWnd;
	return RefreshOptions();
}


////////////////////////////////////////////////////////////////
// CConnectDlg::RefreshOptions
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RefreshOptions()
{
	HWND hWnd = m_hWndOptions;

	//Set CLSCTX to previous saved values
	CheckDlgButton(hWnd, IDB_INPROC_SERVER,			m_dwCLSCTX & CLSCTX_INPROC_SERVER			? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_LOCAL_SERVER,			m_dwCLSCTX & CLSCTX_LOCAL_SERVER			? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_REMOTE_SERVER,			m_dwCLSCTX & CLSCTX_REMOTE_SERVER			? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INPROC_HANDLER,		m_dwCLSCTX & CLSCTX_INPROC_HANDLER			? BST_CHECKED : BST_UNCHECKED);
	wSendMessage(GetDlgItem(hWnd, IDE_REMOTESERVER), WM_SETTEXT, 0, m_wszRemoteServer);
	EnableWindow(GetDlgItem(hWnd, IDE_REMOTESERVER), m_dwCLSCTX & CLSCTX_REMOTE_SERVER);

	//Set Connection Options to previous saved values
	CheckDlgButton(hWnd, IDB_INIT_INITIALIZE,		m_dwConnectOpts & CONNECT_INITIALIZE		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INIT_SETPROPERTIES,	m_dwConnectOpts & CONNECT_SETPROPERTIES		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INIT_CREATESESSION,	m_dwConnectOpts & CONNECT_CREATESESSION		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INIT_CREATECOMMAND,	m_dwConnectOpts & CONNECT_CREATECOMMAND		? BST_CHECKED : BST_UNCHECKED);

	//Set Other Options
	CheckDlgButton(hWnd, IDB_USESERVICECOMP,		m_dwConnectOpts & CONNECT_USESERVICECOMP	? BST_CHECKED : BST_UNCHECKED);
	EnableWindow(GetDlgItem(hWnd, IDB_USESERVICECOMP), (BOOL)pIDataInitialize());
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::UpdateOptions
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::UpdateOptions()
{
	//Obtain CLSCTX
	m_dwCLSCTX = 0;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_INPROC_SERVER))
		m_dwCLSCTX |= CLSCTX_INPROC_SERVER;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_LOCAL_SERVER))
		m_dwCLSCTX |= CLSCTX_LOCAL_SERVER;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_REMOTE_SERVER))
		m_dwCLSCTX |= CLSCTX_REMOTE_SERVER;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_INPROC_HANDLER))
		m_dwCLSCTX |= CLSCTX_INPROC_HANDLER;

	//Obtain RemoteServer
	wSendMessage(GetDlgItem(m_hWndOptions, IDE_REMOTESERVER), WM_GETTEXT, MAX_NAME_LEN, m_wszRemoteServer);

	//Obtain Connection Options
	m_dwConnectOpts = 0;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_INIT_INITIALIZE))
		m_dwConnectOpts |= CONNECT_INITIALIZE;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_INIT_SETPROPERTIES))
		m_dwConnectOpts |= CONNECT_SETPROPERTIES;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_INIT_CREATESESSION))
		m_dwConnectOpts |= CONNECT_CREATESESSION;
	if(IsDlgButtonChecked(m_hWndOptions, IDB_INIT_CREATECOMMAND))
		m_dwConnectOpts |= CONNECT_CREATECOMMAND;

	//Obtain Other Options
	if(IsDlgButtonChecked(m_hWndOptions, IDB_USESERVICECOMP))
		m_dwConnectOpts |= CONNECT_USESERVICECOMP;
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::InitTrace
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::InitTrace(HWND hWnd)
{
	m_hWndTrace = hWnd;
	RECT rect, rectTest;
	GetWindowRect(m_hWndTrace, &rect);
	GetWindowRect(GetDlgItem(m_hWndTrace, IDB_TEST), &rectTest);

	//Change the FONT
	HFONT hFont = (HFONT)SendMessage(m_hWndTrace, WM_GETFONT, 0, 0);
	SendMessage(m_pCListBox->m_hWnd, WM_SETFONT, (WPARAM)hFont, 0);

	//Change the Window Parent of the ListBox to be this one...
	MoveWindow(m_pCListBox->m_hWnd, 0, 0, (rectTest.left-rect.left) - (rect.right-rectTest.right), rect.bottom-rect.top, TRUE);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::RefreshTrace
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::RefreshTrace()
{
	HWND hWnd = m_hWndTrace;
	SetParent(m_pCListBox->m_hWnd, m_hWndTrace);
	ShowWindow(m_pCListBox->m_hWnd, SW_SHOW);
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::UpdateTrace
//
/////////////////////////////////////////////////////////////////
BOOL CConnectDlg::UpdateTrace()
{
	ShowWindow(m_pCListBox->m_hWnd, SW_HIDE);
	SetParent(m_pCListBox->m_hWnd, m_pCMainWindow->m_hWnd);
	return TRUE;
}

////////////////////////////////////////////////////////////////
// CConnectDlg::GetProviderName
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::GetProviderName()
{
	HWND hWndProv		= GetDlgItem(m_hWndProvider, IDC_PROVIDER);

	//Obtain Provider from DropDown
	if(m_pIParseDisplayName == NULL)
	{
		LONG iEnumIndex = 0;
		LONG iProvIndex = SendMessage(hWndProv, CB_GETCURSEL, 0, 0L);
		if(iProvIndex == CB_ERR || !m_pCEnum->IsConnectedToRootEnum())
		{
			//The user may have typed in a new ProviderName (that the RootEnum) didn't
			//Find or isn't registered, or plainly is working correctly...
			//So just get the text the user typed in...
			memset(&m_EnumInfo, 0, sizeof(ENUMINFO));
			wSendMessage(hWndProv, WM_GETTEXT, MAX_NAME_LEN, m_EnumInfo.wszName);
			wSendMessage(hWndProv, WM_GETTEXT, MAX_NAME_LEN, m_EnumInfo.wszParseName);
		}
		else
		{
			//Since we have the CBS_SORT turned on, the order in the Combo Box does
			//not match our array, so we pass the array index (lParam) as the item data
			iEnumIndex = SendMessage(hWndProv, CB_GETITEMDATA, iProvIndex, 0L);
			memcpy(&m_EnumInfo, m_pCEnum->GetEnumInfo(iEnumIndex), sizeof(ENUMINFO));
		}
	}	

	//Display Provider Info (Title of Window)
	wSendMessageFmt(GetParent(m_hWndProvider), WM_SETTEXT, 0, L"Full Connect - %s", m_EnumInfo.wszDescription[0] ? m_EnumInfo.wszDescription : L"No Descripition Avialable");
	return S_OK;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::CreateProviderInstance
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::CreateProviderInstance()
{
	IUnknown* pIUnknown = NULL;
	HRESULT hr = S_OK;

	//Deletgate out the RootEnumerator to instansiate this object
	//This way if the root enum is not installed it will just CoCreate on it
	TESTC(hr = m_pCEnum->CreateProvider(m_pIParseDisplayName, m_EnumInfo.wszParseName, m_dwCLSCTX, IID_IUnknown, &pIUnknown, m_dwConnectOpts, m_wszRemoteServer));
	SAFE_RELEASE(m_pIUnknown);
	m_pIUnknown = pIUnknown;
	
CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::GetPropSets
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::GetPropSets()
{
	HRESULT hr = S_OK;

	//Free Previous Properties
	//This method gets called whenever connect gets called.
	//So the user may or may not have called 'More', which would have already 
	//Obtained these properties.  Also the user may have updated the dialogs
	//afterwards, so bascially it boils down to the fact we need to "reget" all
	//Properties that are in the dialog, and save all the prev properties (More)
	ULONG cAdvPropSets = m_cPropSets;
	DBPROPSET* rgAdvPropSets = m_rgPropSets;
	m_cPropSets = 0;
	m_rgPropSets = NULL;

	//Setup Properties from Provider Page	
	if(m_wszLocation[0])
		SetProperty(DBPROP_INIT_LOCATION,		DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BSTR, m_wszLocation);
	if(m_wszDataSource[0])
		SetProperty(DBPROP_INIT_DATASOURCE,		DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BSTR, m_wszDataSource);
	if(m_wszUserID[0])
		SetProperty(DBPROP_AUTH_USERID,			DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BSTR, m_wszUserID);
	if(m_wszPassword[0])
		SetProperty(DBPROP_AUTH_PASSWORD,		DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BSTR, m_wszPassword);
	if(m_dwPromptProp != NOTSET)
	{
		SetProperty(DBPROP_INIT_PROMPT,			DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I2, &m_dwPromptProp);
		SetProperty(DBPROP_INIT_HWND,			DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I4, &m_dwhWndProp);
	}

	//Setup Properties from Properties Page
	if(m_wszProvString[0])
		SetProperty(DBPROP_INIT_PROVIDERSTRING,	DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BSTR, m_wszProvString);
	if(m_wszCatalog[0])
		SetProperty(DBPROP_INIT_CATALOG,		DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BSTR, m_wszCatalog);
	if(m_dwTimeoutProp != NOTSET)
		SetProperty(DBPROP_INIT_TIMEOUT,		DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I4,	&m_dwTimeoutProp);
	if(m_dwAsynchProp != NOTSET)
		SetProperty(DBPROP_INIT_ASYNCH,			DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I4,	&m_dwAsynchProp);
	if(m_dwlcidProp != NOTSET)
		SetProperty(DBPROP_INIT_LCID,			DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I4,	&m_dwlcidProp);
	if(m_dwModeProp != NOTSET)
		SetProperty(DBPROP_INIT_MODE,			DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I4,	&m_dwModeProp);

	//Setup Properties from Security Page
	if(m_dwProtectionProp != NOTSET)
		SetProperty(DBPROP_INIT_PROTECTION_LEVEL,DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I4,	&m_dwProtectionProp);
	if(m_dwImpersonateProp != NOTSET)
		SetProperty(DBPROP_INIT_IMPERSONATION_LEVEL,DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_I4,	&m_dwImpersonateProp);
	if(m_dwMaskPasswordProp != NOTSET)
		SetProperty(DBPROP_AUTH_MASK_PASSWORD,	DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BOOL,	&m_dwMaskPasswordProp);
	if(m_dwEncryptPasswordProp != NOTSET)
		SetProperty(DBPROP_AUTH_ENCRYPT_PASSWORD,DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BOOL,	&m_dwEncryptPasswordProp);
	if(m_dwCacheProp != NOTSET)
		SetProperty(DBPROP_AUTH_CACHE_AUTHINFO,	DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BOOL,	&m_dwCacheProp);
	if(m_dwPersistProp != NOTSET)
		SetProperty(DBPROP_AUTH_PERSIST_SENSITIVE_AUTHINFO,	DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BOOL,	&m_dwPersistProp);
	if(m_dwPersistEncryptProp != NOTSET)
		SetProperty(DBPROP_AUTH_PERSIST_ENCRYPTED,	DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BOOL,	&m_dwPersistEncryptProp);
	if(m_wszIntegrated[0])
		SetProperty(DBPROP_AUTH_INTEGRATED,		DBPROPSET_DBINIT, &m_cPropSets, &m_rgPropSets, DBTYPE_BSTR, m_wszIntegrated);

	//Loop over all the Advanced Properties set,
	//And adjust any of the existing property options or type.
	//Always use the "latest" value specified in the dialogs, but the Advncaed
	//is the only way to set the type or options (REQUIRED/OPTIONAL)...
	
	//If there are properties that exist out side of the ones avilable in the 
	//dialogs, jusat set it, it may be an provider specific property,
	//or a newly added one to the group...
	for(ULONG i=0; i<cAdvPropSets; i++)
	{
		DBPROPSET* pAdvPropSet = &rgAdvPropSets[i];
		for(ULONG j=0; j<pAdvPropSet->cProperties; j++)
		{
			DBPROP* pAdvProp = &pAdvPropSet->rgProperties[j];
			switch(pAdvProp->dwPropertyID)
			{
				//Provider Page
				case DBPROP_INIT_LOCATION:
				case DBPROP_INIT_DATASOURCE:
				case DBPROP_AUTH_USERID:
				case DBPROP_AUTH_PASSWORD:
				case DBPROP_INIT_PROMPT:
				case DBPROP_INIT_HWND:
				
				//Propeerties Page
				case DBPROP_INIT_PROVIDERSTRING:
				case DBPROP_INIT_CATALOG:
				case DBPROP_INIT_LCID:
				case DBPROP_INIT_ASYNCH:
				case DBPROP_INIT_TIMEOUT:
				case DBPROP_INIT_MODE:
				
				//Security Page
				case DBPROP_INIT_PROTECTION_LEVEL:
				case DBPROP_INIT_IMPERSONATION_LEVEL:
				case DBPROP_AUTH_MASK_PASSWORD:
				case DBPROP_AUTH_ENCRYPT_PASSWORD:
				case DBPROP_AUTH_CACHE_AUTHINFO:
				case DBPROP_AUTH_PERSIST_SENSITIVE_AUTHINFO:
				case DBPROP_AUTH_PERSIST_ENCRYPTED:
				case DBPROP_AUTH_INTEGRATED:
				{	
					//All off these properties are avialable through the Dialogs
					//But not all the options and types are.  So if this property
					//exists in the current set, then adjust any options and types...
					//Since these are only exposed in the Advnaced page...
					
					//Find this property in the current set...
					DBPROP* pPropFound = NULL;
					if(FindProperty(pAdvProp->dwPropertyID, pAdvPropSet->guidPropertySet, m_cPropSets, m_rgPropSets, &pPropFound))
					{
						//Override any options - from advanced
						pPropFound->dwOptions = pAdvProp->dwOptions;
						//Override any wType - from advanced
						pPropFound->vValue.vt = pAdvProp->vValue.vt;
						//Override any colid - from advanced
						//memcpy(pProp->colid, pAdvProp->colid, sizeof(DBID));
					}
					else
					{
						//It may not have been found in the current set
						//since the dialog may have been Empty, ("").  Its hard
						//to determine what empty means in a edit box, does it mean
						//Empty String, NULL, not set, or VT_EMPTY.
						//If this property exists in the Advanced set this is where
						//we will obtain this information...

						//The Advanced Page will always override the edit boxes...
						SetProperty(pAdvProp->dwPropertyID, pAdvPropSet->guidPropertySet, &m_cPropSets, &m_rgPropSets, DBTYPE_VARIANT, &pAdvProp->vValue, pAdvProp->dwOptions);
					}
					break;
				}
				
				default:
				{
					//Otherwise this is not a settable property through the 
					//Dialogs, so just add the property, since it may be a 
					//provider specific property or newly added to DBINIT set...
					SetProperty(pAdvProp->dwPropertyID, pAdvPropSet->guidPropertySet, &m_cPropSets, &m_rgPropSets, DBTYPE_VARIANT, &pAdvProp->vValue, pAdvProp->dwOptions);
					break;
				}
			}
		}
	};

	FreeProperties(&cAdvPropSets, &rgAdvPropSets);
	return hr;
}

/////////////////////////////////////////////////////////////////
// CConnectDlg::FullConnect
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::FullConnect()
{
	HRESULT hr = S_OK;
	CMDIChild* pCMDIChild = NULL;

	//Obtain instance of Provider/Enum (if haven't done so already)...
	TESTC(hr = CreateProviderInstance());

	//Setup PropSets from all saved options
	TESTC(hr = GetPropSets());

	//Connect using the specified properties
	pCMDIChild = new CMDIChild(NULL, m_hInst, m_pCMainWindow);
	TESTC(hr = pCMDIChild->m_pCRowset->CreateConnection(m_pIUnknown, m_cPropSets, m_rgPropSets, m_dwConnectOpts));
	
	//Copy all info to the CListBox
	m_pCListBox->CopyTo(pCMDIChild->m_pCListBox->m_hWnd);

	//SetEnumInfo for UI use...
	pCMDIChild->m_pCRowset->m_pCDataSource->SetEnumInfo(&m_EnumInfo);

	//And Create NewChildWindow
	//Otherwise we are connected, so display ChildWindow
	pCMDIChild->Display();
	
	//Update the Saved Configurations (now that successfuly connected)
	AddRecentConfig();

CLEANUP:
	if(FAILED(hr))
		SAFE_DELETE(pCMDIChild);
	return hr;
}


/////////////////////////////////////////////////////////////////
// CConnectDlg::PromptDataSource
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::PromptDataSource(DBPROMPTOPTIONS dwPromptOptions, REFIID riid)
{
	ASSERT(m_pIDBPromptInitialize);
	HRESULT hr = S_OK;
	BOOL bDisplayed = FALSE;
	HWND hWnd = m_hWnd;
	CMDIChild* pCMDIChild = NULL;
	CListBox* pCListBox = m_pCListBox;
	IUnknown* pIUnknown = NULL;

	//IDBPromptInitalize::PromptDataSource
	TESTC(pCListBox->OutputPreMethod("IDBPromptInitialize::PromptDataSource(0x%08x, %d, 0, NULL, NULL, %s, &0x%08x)", hWnd, dwPromptOptions, GetInterfaceName(riid), pIUnknown));
	XTEST(hWnd, hr = m_pIDBPromptInitialize->PromptDataSource
				(
				NULL,
				hWnd,								// hWndParent
				dwPromptOptions,					// dwPromptOptions
            	0,									// cSourceTypeFilter
            	NULL,								// rgSourceTypeFilter
            	NULL,								// pwszszzProviderFilter
            	riid,								// riid
            	(IUnknown **)&pIUnknown				// ppDataSource
				));
	TESTC(pCListBox->OutputPostMethod(hr, "IDBPromptInitialize::PromptDataSource(0x%08x, %d, 0, NULL, NULL, %s, &0x%08x)", hWnd, dwPromptOptions, GetInterfaceName(riid), pIUnknown));
				

	//Connect - DSL should have already set the properties 
	pCMDIChild = new CMDIChild(NULL, m_hInst, m_pCMainWindow);
	TESTC(hr = pCMDIChild->m_pCRowset->CreateConnection(pIUnknown, 0, NULL, m_dwConnectOpts));
	
	//Copy all info to the CListBox
	m_pCListBox->CopyTo(pCMDIChild->m_pCListBox->m_hWnd);

	//And Create NewChildWindow
	//Otherwise we are connected, so display ChildWindow
	pCMDIChild->Display();
	bDisplayed = TRUE;
	
CLEANUP:
	if(!bDisplayed)
		SAFE_DELETE(pCMDIChild);
	SAFE_RELEASE(pIUnknown);
	return hr;
}


/////////////////////////////////////////////////////////////////
// CConnectDlg::PromptFileName
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::PromptFileName()
{
	ASSERT(m_pIDBPromptInitialize);
	HRESULT hr = S_OK;
	BOOL bDisplayed = FALSE;
	HWND hWnd = m_hWnd;
	CMDIChild* pCMDIChild = NULL;
	CListBox* pCListBox = m_pCListBox;
	WCHAR* pwszSelectedFile = NULL;
	WCHAR* pwszInitString = NULL;
	IUnknown* pIUnknown = NULL;

	//IDBPromptInitalize::PromptFileName
	TESTC(pCListBox->OutputPreMethod("IDBPromptInitialize::PromptFileName(0x%08x, DBPROMPTOPTIONS_NONE, NULL, NULL, &%S)", hWnd, pwszSelectedFile));
	XTEST(hWnd, hr = m_pIDBPromptInitialize->PromptFileName
				(
				hWnd,								// hWndParent
				DBPROMPTOPTIONS_NONE,				// dwPromptOptions
            	NULL,								// pwszInitialDirectory
            	NULL,								// pwszInitialFile
            	&pwszSelectedFile					// pwszSelectedFile
				));
	TESTC(pCListBox->OutputPostMethod(hr, "IDBPromptInitialize::PromptFileName(0x%08x, DBPROMPTOPTIONS_NONE, NULL, NULL, &%S)", hWnd, pwszSelectedFile));
	//Delegate, and Connect from the Selected File
	TESTC(hr = ConnectFromFile(pwszSelectedFile));

CLEANUP:
	SAFE_FREE(pwszSelectedFile);
	SAFE_RELEASE(pIUnknown);
	return hr;
}


/////////////////////////////////////////////////////////////////
// CConnectDlg::GetInitString
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::GetInitString(IUnknown* pIUnknown, BOOL fIncludePassword, WCHAR** ppwszInitString)
{
	HRESULT hr = S_OK;
	WCHAR* pwszInitString = NULL;
	CListBox* pCListBox = m_pCListBox;
	HWND hWnd = m_hWnd;

	//Obtain the correct ListBox
	CMDIChild* pCMDIChild = m_pCMainWindow->GetActiveChildObj();
	if(pCMDIChild)
	{
		hWnd = pCMDIChild->m_hWnd;
		pCListBox = pCMDIChild->m_pCListBox;
	}

	//GetInitializationString
	TESTC(pCListBox->OutputPreMethod("IDataInitialize::GetInitializationString(0x%08x, %s, &\"%S\")", pIUnknown, fIncludePassword ? "True" : "False", pwszInitString));
	XTEST(hWnd, hr = pIDataInitialize()->GetInitializationString(pIUnknown, fIncludePassword, &pwszInitString));
	TESTC(pCListBox->OutputPostMethod(hr, "IDataInitialize::GetInitializationString(0x%08x, %s, &\"%S\")", pIUnknown, fIncludePassword ? "True" : "False", pwszInitString));

CLEANUP:
	if(ppwszInitString)
		*ppwszInitString = pwszInitString;
	else
		SAFE_FREE(pwszInitString);
	return hr;
}

/////////////////////////////////////////////////////////////////
// CConnectDlg::SaveInitString
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::SaveInitString(WCHAR* pwszFileName, WCHAR* pwszInitString, DWORD dwCreateOpts)
{
	HRESULT hr = S_OK;
	HWND hWnd = m_hWnd;
	CListBox* pCListBox = m_pCListBox;

	//Obtain the correct ListBox
	CMDIChild* pCMDIChild = m_pCMainWindow->GetActiveChildObj();
	if(pCMDIChild)
	{
		hWnd = pCMDIChild->m_hWnd;
		pCListBox = pCMDIChild->m_pCListBox;
	}

	//WriteStringToStorage
	TESTC(pCListBox->OutputPreMethod("IDataInitialize::WriteStringToStorage(\"%S\", \"%S\", 0x%08x)", pwszFileName, pwszInitString, dwCreateOpts));
	XTEST(hWnd, hr = pIDataInitialize()->WriteStringToStorage(pwszFileName, pwszInitString, dwCreateOpts));
	TESTC(pCListBox->OutputPostMethod(hr, "IDataInitialize::WriteStringToStorage(\"%S\", \"%S\", 0x%08x)", pwszFileName, pwszInitString, dwCreateOpts));

CLEANUP:
	return hr;
}

	
/////////////////////////////////////////////////////////////////
// CConnectDlg::LoadInitString
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::LoadInitString(WCHAR* pwszFileName, WCHAR** ppwszInitString)
{
	HRESULT hr = S_OK;
	HWND hWnd = m_hWnd;
	CListBox* pCListBox = m_pCListBox;
	WCHAR* pwszInitString = NULL;

	//Obtain the correct ListBox
	CMDIChild* pCMDIChild = m_pCMainWindow->GetActiveChildObj();
	if(pCMDIChild)
	{
		hWnd = pCMDIChild->m_hWnd;
		pCListBox = pCMDIChild->m_pCListBox;
	}

	//LoadStringFromStorage
	TESTC(pCListBox->OutputPreMethod("IDataInitialize::LoadStringFromStorage(\"%S\", &\"%S\")", pwszFileName, pwszInitString));
	XTEST(hWnd, hr = pIDataInitialize()->LoadStringFromStorage(pwszFileName, &pwszInitString));
	TESTC(pCListBox->OutputPostMethod(hr, "IDataInitialize::LoadStringFromStorage(\"%S\", &\"%S\")", pwszFileName, pwszInitString));

CLEANUP:
	if(ppwszInitString)
		*ppwszInitString = pwszInitString;
	else
		SAFE_FREE(pwszInitString);
	return hr;
}


/////////////////////////////////////////////////////////////////
// CConnectDlg::ConnectFromFile
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::ConnectFromFile(WCHAR* pwszSelectedFile, IUnknown* pIUnkOuter, REFIID riid)
{
	HRESULT hr = S_OK;
	BOOL bDisplayed = FALSE;

	//No-op
	if(pwszSelectedFile == NULL)
		return E_FAIL;

	HWND hWnd = m_hWnd;
	CMDIChild* pCMDIChild = NULL;
	CHAR szBuffer[MAX_NAME_LEN+1];
	WCHAR* pwszInitString = NULL;

	//Load the saved InitString from the SelectedFile
	TESTC(hr = LoadInitString(pwszSelectedFile, &pwszInitString));

	//Deletgate - Now that we have the InitString
	TESTC(hr = ConnectFromString(pwszInitString, pIUnkOuter, riid));

	//Update the Saved Files (now that successfuly connected)
	ConvertToMBCS(pwszSelectedFile, szBuffer, MAX_NAME_LEN);
	AddRecentFile(szBuffer);

CLEANUP:
	SAFE_FREE(pwszInitString);
	return hr;
}


////////////////////////////////////////////////////////////////
// CConnectDlg::ConnectFromString
//
/////////////////////////////////////////////////////////////////
HRESULT CConnectDlg::ConnectFromString(WCHAR* pwszInitString, IUnknown* pIUnkOuter, REFIID riid, IUnknown* pExistingDataSource)
{
	HRESULT hr = S_OK;
	BOOL bDisplayed = FALSE;
	HWND hWnd = m_hWnd;
	CMDIChild* pCMDIChild = NULL;
	CListBox* pCListBox = m_pCListBox;
	IUnknown* pIUnknown = pExistingDataSource;
	
	//GetDataSource based upoon the InitString
	TESTC(pCListBox->OutputPreMethod("IDataInitialize::GetDataSource(0x%08x, 0x%08x, \"%S\", %s, &0x%08x)", pIUnkOuter, GetOptionsObj()->m_dwDSOCLSCTX, pwszInitString, GetInterfaceName(riid), pIUnknown));
	XTEST(hWnd, hr = pIDataInitialize()->GetDataSource(pIUnkOuter, GetOptionsObj()->m_dwDSOCLSCTX, pwszInitString, riid, &pIUnknown));
	TESTC(pCListBox->OutputPostMethod(hr, "IDataInitialize::GetDataSource(0x%08x, 0x%08x, \"%S\", %s, &0x%08x)", pIUnkOuter, GetOptionsObj()->m_dwDSOCLSCTX, pwszInitString, GetInterfaceName(riid), pIUnknown));

	//Connect - DSL should have already set the properties 
	pCMDIChild = new CMDIChild(NULL, m_hInst, m_pCMainWindow);
	TESTC(hr = pCMDIChild->m_pCRowset->CreateConnection(pIUnknown, 0, NULL, GetOptionsObj()->m_dwDataSourceOpts));
	
	//Copy all info to the CListBox
	pCListBox->CopyTo(pCMDIChild->m_pCListBox->m_hWnd);

	//SetEnumInfo for UI use...
	pCMDIChild->m_pCRowset->m_pCDataSource->SetEnumInfo(NULL);

	//And Create NewChildWindow
	//Otherwise we are connected, so display ChildWindow
	pCMDIChild->Display();
	bDisplayed = TRUE;

CLEANUP:
	if(!bDisplayed)
		SAFE_DELETE(pCMDIChild);
	SAFE_RELEASE(pIUnknown);
	return hr;
}


////////////////////////////////////////////////////////////////
// CPropDlg::CPropDlg
//
/////////////////////////////////////////////////////////////////
CPropDlg::CPropDlg(HWND hWnd, HINSTANCE hInst, CMainWindow* pCMainWindow, CListBox* pCListBox)
	: CDialogBase(hWnd, hInst)
{
	ASSERT(pCMainWindow);
	m_pCMainWindow = pCMainWindow;
	ASSERT(pCListBox);
	m_pCListBox	= pCListBox;

	m_eSource			= ROWSET;
	m_pIUnkProperties	= NULL;
	m_pIUnkPropInfo		= NULL;

	m_pcPropSets		= 0;
	m_prgPropSets		= NULL;
}


////////////////////////////////////////////////////////////////
// CPropDlg::~CPropDlg
//
/////////////////////////////////////////////////////////////////
CPropDlg::~CPropDlg()
{
}


////////////////////////////////////////////////////////////////
// CPropDlg::InitControls
//
/////////////////////////////////////////////////////////////////
BOOL CPropDlg::InitControls(HWND hWnd)
{
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CPropDlg::RefreshControls
//
/////////////////////////////////////////////////////////////////
BOOL CPropDlg::RefreshControls()
{
	return TRUE;
}


////////////////////////////////////////////////////////////////
// CPropDlg::SetProperties
//
/////////////////////////////////////////////////////////////////
HRESULT CPropDlg::SetProperties(HWND hWnd, SOURCE eSource, IUnknown* pIUnkProperties, IUnknown* pIUnkPropInfo, ULONG* pcPropSets, DBPROPSET** prgPropSets)
{
	//Setup variable for StaticProcs to use...
	m_eSource			= eSource;
	m_pIUnkProperties	= pIUnkProperties;
	m_pIUnkPropInfo		= pIUnkPropInfo;

	m_pcPropSets		= pcPropSets;
	m_prgPropSets		= prgPropSets;

	//Now Display the dialog
	return DialogBoxParam(m_hInst, MAKEINTRESOURCE(IDD_SETPROPERTIES), hWnd, SetPropertiesProc, (LPARAM)this);
}


////////////////////////////////////////////////////////////////
// CPropDlg::GetProperties
//
/////////////////////////////////////////////////////////////////
HRESULT CPropDlg::GetProperties(HWND hWnd, SOURCE eSource, IUnknown* pIUnkProperties, ULONG* pcPropSets, DBPROPSET** prgPropSets)
{
	ASSERT(pIUnkProperties);

	//Setup variable for StaticProcs to use...
	m_eSource			= eSource;
	m_pIUnkProperties	= pIUnkProperties;
	m_pIUnkPropInfo		= NULL;

	m_pcPropSets		= pcPropSets;
	m_prgPropSets		= prgPropSets;

	//Now Display the dialog
	return DialogBoxParam(m_hInst, MAKEINTRESOURCE(IDD_GETPROPERTIES), hWnd, GetPropertiesProc, (LPARAM)this);
}


////////////////////////////////////////////////////////////////
// CPropDlg::GetPropertyInfo
//
/////////////////////////////////////////////////////////////////
HRESULT CPropDlg::GetPropertyInfo(HWND hWnd, SOURCE eSource, IUnknown* pIUnkPropInfo)
{
	ASSERT(pIUnkPropInfo);

	//Setup variable for StaticProcs to use...
	m_eSource			= eSource;
	m_pIUnkProperties	= NULL;
	m_pIUnkPropInfo		= pIUnkPropInfo;

	m_pcPropSets		= NULL;
	m_prgPropSets		= NULL;

	//Now Display the dialog
	return DialogBoxParam(m_hInst, MAKEINTRESOURCE(IDD_GETPROPERTIES), hWnd, GetPropertyInfoProc, (LPARAM)this);
}


////////////////////////////////////////////////////////////////
// CPropDlg::DisplaySetProperties
//
/////////////////////////////////////////////////////////////////
HRESULT CPropDlg::DisplaySetProperties(HWND hWnd, ULONG* pcPropInfoSets, DBPROPINFOSET** prgPropInfoSets, WCHAR** ppStringBuffer, GUID guidPropertySet)
{
	ASSERT(pcPropInfoSets);
	ASSERT(prgPropInfoSets);
	ASSERT(ppStringBuffer);

	HRESULT hr = S_OK;
	HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
	CListBox* pCListBox = m_pCListBox;
	ULONG i,j,iItems = 0;
	CHAR szBuffer[MAX_NAME_LEN+1];
	WCHAR wszBuffer[MAX_NAME_LEN+1];

	DBPROPIDSET PropertyIDSet[1];
	PropertyIDSet[0].cPropertyIDs = 0;
	PropertyIDSet[0].rgPropertyIDs = NULL;
	PropertyIDSet[0].guidPropertySet = guidPropertySet;
	ULONG cPropertyIDSets = (guidPropertySet == IID_NULL ? 0 : 1);
	DBPROPIDSET* rgPropertyIDSets = cPropertyIDSets==0 ? NULL : PropertyIDSet;
 
	ULONG cPropSets = 0;
	DBPROPSET* rgPropSets = NULL;

	ULONG cPropInfoSets = 0;
	DBPROPINFOSET* rgPropInfoSets = NULL;
	WCHAR* pStringBuffer = NULL;
	BOOL fSetProps = m_pcPropSets && m_prgPropSets;

	//The only way we can populate the SetProperties dialog is with
	//GetProeprtyInfo, if not available we just have to exit...
	if(m_pIUnkPropInfo == NULL)
	{
		hr = E_FAIL;
		goto CLEANUP;
	}

	//If using Properties passed in
	if(fSetProps)
	{
		cPropSets = *(m_pcPropSets);
		rgPropSets = *(m_prgPropSets);
	}
	else
	{
		switch(m_eSource)
		{
			case DATASOURCEINIT:
				//IDBProperties::GetProperties
				ASSERT(m_pIUnkProperties);
				TESTC(pCListBox->OutputPreMethod("IDBProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				XTEST(hWnd, hr = ((IDBProperties*)m_pIUnkProperties)->GetProperties(0, NULL, &cPropSets, &rgPropSets));
				TESTC(pCListBox->OutputPostMethod(hr, "IDBProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				break;

			case DATASOURCE:
				//IDBProperties::GetProperties
				ASSERT(m_pIUnkProperties);
				TESTC(pCListBox->OutputPreMethod("IDBProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				XTEST(hWnd, hr = ((IDBProperties*)m_pIUnkProperties)->GetProperties(0, NULL, &cPropSets, &rgPropSets));
				TESTC(pCListBox->OutputPostMethod(hr, "IDBProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				break;

			case SESSION:
				//ISessionProperties::GetProperties
				ASSERT(m_pIUnkProperties);
				TESTC(pCListBox->OutputPreMethod("ISessionProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				XTEST(hWnd, hr = ((ISessionProperties*)m_pIUnkProperties)->GetProperties(0, NULL, &cPropSets, &rgPropSets));
				TESTC(pCListBox->OutputPostMethod(hr, "ISessionProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				break;

			case COMMAND:
				//ISessionProperties::GetProperties
				ASSERT(m_pIUnkProperties);
				TESTC(pCListBox->OutputPreMethod("ICommandProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				XTEST(hWnd, hr = ((ICommandProperties*)m_pIUnkProperties)->GetProperties(0, NULL, &cPropSets, &rgPropSets));
				TESTC(pCListBox->OutputPostMethod(hr, "ICommandProperties::GetProperties(0, NULL, &%d, &0x%08x)", cPropSets, rgPropSets));
				break;
		};
	}

	//Obtain all the properties...
	//Normally GetPropertyInfo supplies all the Property Information,
	//But OLE DB does have a couple of other interfaces that supply prop info
	switch(m_eSource)
	{
		case DATASOURCEADMIN:
			//IDBDataSourceAdmin::GetCreationProperties
			TESTC(pCListBox->OutputPreMethod("IDBDataSourceAdmin::GetCreationProperties(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			XTEST(hWnd, hr = ((IDBDataSourceAdmin*)m_pIUnkPropInfo)->GetCreationProperties(cPropertyIDSets, rgPropertyIDSets, &cPropInfoSets, &rgPropInfoSets, &pStringBuffer));
			TESTC(pCListBox->OutputPostMethod(hr, "IDBDataSourceAdmin::GetCreationProperties(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			break;

		default:
			//IDBProperties::GetPropertyInfo
			TESTC(pCListBox->OutputPreMethod("IDBProperties::GetPropertyInfo(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			XTEST(hWnd, hr = ((IDBProperties*)m_pIUnkPropInfo)->GetPropertyInfo(cPropertyIDSets, rgPropertyIDSets, &cPropInfoSets, &rgPropInfoSets, &pStringBuffer));
			TESTC(pCListBox->OutputPostMethod(hr, "IDBProperties::GetPropertyInfo(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			break;

	};

	//Display PropertyNames
	iItems = 0;
	for(i=0; i<cPropInfoSets; i++)
	{
		DBPROPINFOSET* pPropInfoSet = &rgPropInfoSets[i];
		
		//We need to display this PropertySet
		CHAR* pszName = GetPropSetName(pPropInfoSet->guidPropertySet);
		if(pszName == NULL)
		{
			StringFromGUID2(pPropInfoSet->guidPropertySet, wszBuffer, MAX_NAME_LEN);
			ConvertToMBCS(wszBuffer, szBuffer, MAX_NAME_LEN);
			pszName = szBuffer;
		}
		LV_InsertItem(hWndPropNames, iItems, PROP_NAME, pszName, PARAM_NONE, IMAGE_NORMAL);
		LV_SetItemState(hWndPropNames, iItems, 0,  STATE_NORMAL,  LVIS_STATEIMAGEMASK);
		iItems++;

		//Now display all properties in the set
		for(j=0; j<rgPropInfoSets[i].cPropertyInfos; j++)
		{
			DBPROPINFO* pPropInfo = &pPropInfoSet->rgPropertyInfos[j];
			DBPROP* pFoundProp = NULL;

			//Try to find this Property in the PropSet
			FindProperty(pPropInfo->dwPropertyID, pPropInfoSet->guidPropertySet, cPropSets, rgPropSets, &pFoundProp);
			VARIANT* pVariant = pFoundProp ? &pFoundProp->vValue : &pPropInfo->vValues;

			//Determine which icon to display
			LONG iImage = pPropInfo->dwFlags & DBPROPFLAGS_WRITE ? IMAGE_NORMAL : IMAGE_READONLY;

			//To make our lives easier we will store the [i,j] index into the
			//PropInfoSet or every property into the lParam element of the list view
			//This way whatever message were one we have the neccessary propinfo...
			LONG lParam = MAKELONG(i,j);

			//PROP_NAME (Indent from PropertySet)
			//Also store the wType as the lParam
			pszName = GetPropertyName(pPropInfo->dwPropertyID, pPropInfoSet->guidPropertySet);
			if(pszName)
				sprintf(szBuffer, "     %s", pszName);
			else
				sprintf(szBuffer, "     0x%x", pPropInfo->dwPropertyID);
			LV_InsertItem(hWndPropNames, iItems, PROP_NAME, szBuffer, lParam, iImage);
			
			//Supply Default Type and Values if there is currently none
			if(V_VT(pVariant) == VT_EMPTY && pFoundProp==NULL)
			{
				V_VT(pVariant) = pPropInfo->vtType;
				switch(V_VT(pVariant))
				{
					case VT_EMPTY:
					case VT_NULL:
						break;

					case VT_BOOL:
						V_BOOL(pVariant) = VARIANT_TRUE;
						break;

					case VT_I4:
						V_I4(pVariant) = 0;
						break;

					case VT_I2:
						V_I2(pVariant) = 0;
						break;

					case VT_BSTR:
						V_BSTR(pVariant) = NULL;
						break;

					default:
						V_VT(pVariant) = VT_EMPTY;
						break;
				};

			}
			
			//dwOptions
			pPropInfo->dwFlags = pFoundProp ? pFoundProp->dwOptions : 0;

			//May need to adjust the Variant
			if(pFoundProp)
			{
				pPropInfo->vtType = V_VT(&pFoundProp->vValue);
				XTEST(hWnd, VariantClear(&pPropInfo->vValues));
				XTEST(hWnd, VariantCopy(&pPropInfo->vValues, &pFoundProp->vValue));
			}

			//PROP_TYPE
			LV_InsertItem(hWndPropNames, iItems, PROP_TYPE, GetVariantTypeName(V_VT(pVariant)));

			//PROP_VALUE
			VariantToString(pVariant, szBuffer, MAX_NAME_LEN, CONV_VARBOOL);
			LV_InsertItem(hWndPropNames, iItems, PROP_VALUE, szBuffer);

			//PROP_OPTIONS
			if(pPropInfo->dwFlags == DBPROPOPTIONS_OPTIONAL)
				LV_InsertItem(hWndPropNames, iItems, PROP_OPTIONS, "OPTIONAL");
			else
				LV_InsertItem(hWndPropNames, iItems, PROP_OPTIONS, "REQUIRED");

			//PROP_COLID
			DBIDToString(pFoundProp ? &pFoundProp->colid : NULL, wszBuffer, MAX_NAME_LEN);
			ConvertToMBCS(wszBuffer, szBuffer, MAX_NAME_LEN);
			LV_InsertItem(hWndPropNames, iItems, PROP_COLID, szBuffer);

			//PROP_DESC
			TESTC(ConvertToMBCS(pPropInfo->pwszDescription, szBuffer, MAX_NAME_LEN));
			LV_InsertItem(hWndPropNames, iItems, PROP_DESC, szBuffer);
			
			//Set Item State to Checked/Unchecked
			LV_SetItemState(hWndPropNames, iItems, 0,  INDEXTOSTATEIMAGEMASK((pFoundProp && fSetProps) ? STATE_CHECKED : STATE_UNCHECKED),  LVIS_STATEIMAGEMASK);
			iItems++;
		}

		//Now display a space between sets...
		LV_InsertItem(hWndPropNames, iItems, 0, "", PARAM_NONE, IMAGE_NORMAL);
		LV_SetItemState(hWndPropNames, iItems, 0,  STATE_NORMAL,  LVIS_STATEIMAGEMASK);
		iItems++;
	}

	//AutoSize Columns
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_NAME,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_TYPE,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_VALUE,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_OPTIONS,(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_DESC,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);

CLEANUP:
	*pcPropInfoSets = cPropInfoSets;
	*prgPropInfoSets = rgPropInfoSets;
	*ppStringBuffer = pStringBuffer;

	if(!fSetProps)
		FreeProperties(&cPropSets, &rgPropSets);
	return hr;
}
			

////////////////////////////////////////////////////////////////
// CPropDlg::SetPropertiesProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CPropDlg::SetPropertiesProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	//NOTE:  These are static variables, just to hold the PropertyInfo
	//Information temporarily.  There are messages that need this info which 
	//gets setup on INITDIALOG.  On OK or CANCEL this info will be freed...
	static ULONG  cPropInfoSets = 0;
	static DBPROPINFOSET* rgPropInfoSets = NULL;
	static WCHAR* pStringBuffer = NULL;
	static GUID   guidPropertySet = IID_NULL;

	switch(message)
	{
		case WM_INITDIALOG:	
		{	
			EXC_BEGIN

			//Save the "this" pointer
			Busy();
			CPropDlg* pThis = (CPropDlg*)SetThis(hWnd, (LPARAM)lParam);

			//Use Extended ListView Styles!
			HWND hWndPropNames			= GetDlgItem(hWnd, IDL_PROPERTY);
			HWND hWndPropSet			= GetDlgItem(hWnd, IDC_PROPSET);
			HWND hWndPropEditOptions	= GetDlgItem(hWnd, IDC_PROPEDIT_OPTIONS);
			HWND hWndPropEditType		= GetDlgItem(hWnd, IDC_PROPEDIT_TYPE);
			SendMessage(hWndPropNames, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES | LVS_EX_CHECKBOXES, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES | LVS_EX_CHECKBOXES);

			//ColumnHeaders
			LV_InsertColumn(hWndPropNames,   PROP_NAME,		"Property");
			LV_InsertColumn(hWndPropNames,   PROP_TYPE,		"Type");
			LV_InsertColumn(hWndPropNames,   PROP_VALUE,	"Value");
			LV_InsertColumn(hWndPropNames,   PROP_OPTIONS,  "Options");
			LV_InsertColumn(hWndPropNames,   PROP_COLID,	"ColID");
			LV_InsertColumn(hWndPropNames,   PROP_DESC,		"Description");
			
			//Setup ImageList
			HIMAGELIST hPropImageList = ImageList_Create(16, 16, ILC_MASK, 1, 0 );
			HIMAGELIST hCheckImageList = ImageList_Create(16, 16, ILC_MASK, 1, 0 );
			
			//IDI_NORMAL
			HICON hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_NORMAL));
			ImageList_AddIcon(hPropImageList, hIcon);
			//IDI_READONLY
			hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_READONLY));
			ImageList_AddIcon(hPropImageList, hIcon);

			//IDI_CHECK - normal checked icon
			hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_CHECK));
			ImageList_AddIcon(hCheckImageList, hIcon);
			//IDI_UNCHECK - normal unchecked icon
			hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_UNCHECK));
			ImageList_AddIcon(hCheckImageList, hIcon);

			//Set image list to the ListView
			ListView_SetImageList(hWndPropNames, hPropImageList, LVSIL_SMALL);
			ListView_SetImageList(hWndPropNames, hCheckImageList, LVSIL_STATE);
//			ImageList_Destroy(hPropImageList);
//			ImageList_Destroy(hCheckImageList);

			//Now adjust display and find required PropSet
			switch(pThis->m_eSource)
			{
				case DATASOURCEINIT:
				{	
					guidPropertySet = DBPROPSET_DBINITALL;
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"IDBProperties::SetProperties");
					break;
				}
				case DATASOURCE:
				{	
					guidPropertySet = DBPROPSET_DATASOURCEALL;
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"IDBProperties::SetProperties");
					break;
				}
				case DATASOURCEADMIN:
				{	
					guidPropertySet = DBPROPSET_DBINITALL;
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"IDBDataSourceAdmin::ModifyDataSource");
					break;
				}
				case SESSION:
				{	
					guidPropertySet = DBPROPSET_SESSIONALL;
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"ISessionProperties::SetProperties");
					break;
				}
				case COMMAND:
				{	
					guidPropertySet = DBPROPSET_ROWSETALL;
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"ICommandProperties::SetProperties");
					break;
				}
				case ROWSET:
				{	
					guidPropertySet = DBPROPSET_ROWSETALL;
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"Set Properties for rgPropSet params");
					break;
				}

				default:
					ASSERT(!"Unhandled Property Source");
					return FALSE;
			};
			
			//Fill in the PROPSET Combo
			for(ULONG i=0; i<g_cPropSetMaps; i++)
			{
				LONG iSel = SendMessage(hWndPropSet, CB_ADDSTRING, 0, (LPARAM)g_rgPropSetMap[i].pszName);
				SendMessage(hWndPropSet, CB_SETITEMDATA, iSel, (LPARAM)i);
			
				if(guidPropertySet == *(g_rgPropSetMap[i].pGuid))
					SendMessage(hWndPropSet, CB_SETCURSEL, iSel, 0);
			}

			//Add DBPROPSET All (0, NULL)...
			LONG iSel = SendMessage(hWndPropSet, CB_ADDSTRING, 0, (LPARAM)"All Properties");
			SendMessage(hWndPropSet, CB_SETITEMDATA, iSel, (LPARAM)g_cPropSetMaps);

			//Default Selection
			iSel = SendMessage(hWndPropSet, CB_GETCURSEL, 0, 0);
			SendMessage(hWndPropSet, CB_SETCURSEL, iSel==CB_ERR || iSel>(LONG)g_cPropSetMaps ? g_cPropSetMaps : iSel, 0);
			
			//Fill in Types Combo
			for(i=0; i<g_cVariantTypes; i++)
			{
				CHAR* pszTypeName = g_rgVariantTypes[i].pszName;
				DBTYPE wType = (DBTYPE)g_rgVariantTypes[i].lItem;
				LONG iSel = SendMessage(hWndPropEditType,	CB_ADDSTRING,	0, (LPARAM)pszTypeName);
				SendMessage(hWndPropEditType,	CB_SETITEMDATA,	iSel, (LPARAM)wType);
			}

			//Fill in Options Combo
			SendMessage(hWndPropEditOptions, CB_ADDSTRING,	0, (LPARAM)"REQUIRED"); // == 0
			SendMessage(hWndPropEditOptions, CB_ADDSTRING,	0, (LPARAM)"OPTIONAL");	// == 1

			//Display the SetProperties (again - this  time all of them)
			TESTC(pThis->DisplaySetProperties(hWnd, &cPropInfoSets, &rgPropInfoSets, &pStringBuffer, guidPropertySet));
			
		CLEANUP:
			//Placement of Dialog, want it just below the "Init" button...
			CenterDialog(hWnd);
			Busy(OFF);

			//Disable Combos
			EnableWindow(GetDlgItem(hWnd, IDE_PROPEDIT_NAME), FALSE);
			EnableWindow(GetDlgItem(hWnd, IDC_PROPEDIT_OPTIONS), FALSE);
			EnableWindow(GetDlgItem(hWnd, IDC_PROPEDIT_TYPE), FALSE);
			EnableWindow(GetDlgItem(hWnd, IDC_PROPEDIT_VALUE), FALSE);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}
		
		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//LBN_SELCHANGE ListBox Selection change
			switch(GET_WM_COMMAND_CMD(wParam, lParam))
			{
				//Selection change in a list box occurred
				case LBN_SELCHANGE:
				{	
					//Otherwise see which combo box has changed
					switch(GET_WM_COMMAND_ID(wParam, lParam))
					{
						case IDC_PROPSET:
						{	
							//Get the "this" pointer
							CPropDlg* pThis = (CPropDlg*)GetThis(hWnd);
							HWND hWndPropSet = GetDlgItem(hWnd, IDC_PROPSET);
							HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
							Busy();

							//Get new selection
							LONG iSel = SendMessage(hWndPropSet, CB_GETCURSEL, 0, 0L);
							LONG iIndex = SendMessage(hWndPropSet, CB_GETITEMDATA, iSel, 0);
							
							guidPropertySet = IID_NULL;
							if(iIndex != CB_ERR && iIndex <(LONG)g_cPropSetMaps)
								guidPropertySet = *(g_rgPropSetMap[iIndex].pGuid); 
							
							//Free any previous properties
							FreeProperties(&cPropInfoSets, &rgPropInfoSets);
							SAFE_FREE(pStringBuffer);

							//Now remove any previous items
							SendMessage(hWndPropNames, LVM_DELETEALLITEMS, 0, 0);

							//Display the SetProperties (again - this  time all of them)
							EXC_TEST(pThis->DisplaySetProperties(hWnd, &cPropInfoSets, &rgPropInfoSets, &pStringBuffer, guidPropertySet));
								
							//Placement of Dialog, want it just below the "Init" button...
							CenterDialog(hWnd);
							Busy(OFF);

							//Disable Combos
							EnableWindow(GetDlgItem(hWnd, IDE_PROPEDIT_NAME), FALSE);
							EnableWindow(GetDlgItem(hWnd, IDC_PROPEDIT_OPTIONS), FALSE);
							EnableWindow(GetDlgItem(hWnd, IDC_PROPEDIT_TYPE), FALSE);
							EnableWindow(GetDlgItem(hWnd, IDC_PROPEDIT_VALUE), FALSE);
							return 0;
						}
					}

					HWND hWndPropNames			= GetDlgItem(hWnd, IDL_PROPERTY);
					HWND hWndPropEditOptions	= GetDlgItem(hWnd, IDC_PROPEDIT_OPTIONS);
					HWND hWndPropEditType		= GetDlgItem(hWnd, IDC_PROPEDIT_TYPE);
					HWND hWndPropEditValue		= GetDlgItem(hWnd, IDC_PROPEDIT_VALUE);

					//Obtain corresponding row in the ListVIew
					//This value is stored as the lParam of the combos...
					LONG iSelRow = SendMessage(hWndPropEditOptions, CB_GETITEMDATA, 0, 0L);
					if(iSelRow == CB_ERR)
						return FALSE;
					
					//Obtain PropInfo element
					//This value is stored as the lParam of the ListView row
					LONG iIndexes = LV_GetItemParam(hWndPropNames, iSelRow, 0);
					if(iIndexes == LVM_ERR)
						return FALSE;
					
					ULONG iPropInfoSet = LOWORD(iIndexes);
					ULONG iPropInfo = HIWORD(iIndexes);
					
					//Verify results...
					if(iPropInfoSet >= cPropInfoSets || rgPropInfoSets==NULL || iPropInfo >= rgPropInfoSets[iPropInfoSet].cPropertyInfos || rgPropInfoSets[iPropInfoSet].rgPropertyInfos==NULL)
						return FALSE;
					
					//Otherwise we have a valid propset and property...
					DBPROPINFO* pPropInfo = &rgPropInfoSets[iPropInfoSet].rgPropertyInfos[iPropInfo];

					//Otherwise see which combo box has changed
					switch(GET_WM_COMMAND_ID(wParam, lParam))
					{
						case IDC_PROPEDIT_OPTIONS:
						{
							//Get the new selection
							pPropInfo->dwFlags = SendMessage(hWndPropEditOptions, CB_GETCURSEL, 0, 0L);
							
							//Display the new selection
							LV_SetItemText(hWndPropNames, iSelRow, PROP_OPTIONS, pPropInfo->dwFlags == DBPROPOPTIONS_REQUIRED ? "REQUIRED" : "OPTIONAL");
							break;
						}
						
						case IDC_PROPEDIT_TYPE:
						{	
							//Get new selection
							LONG iSel = SendMessage(hWndPropEditType, CB_GETCURSEL, 0, 0L);
							pPropInfo->vtType = (DBTYPE)SendMessage(hWndPropEditType, CB_GETITEMDATA, iSel, 0L);
							LV_SetItemText(hWndPropNames, iSelRow, PROP_TYPE, GetVariantTypeName(pPropInfo->vtType));
							
							switch(pPropInfo->vtType)
							{
								case VT_BOOL:
									CB_SelectText(hWndPropEditValue, "VARIANT_FALSE", TRUE);
									CB_SelectText(hWndPropEditValue, "VARIANT_TRUE", TRUE);
									break;

								case VT_EMPTY:
								case VT_NULL:
									SendMessage(hWndPropEditValue, CB_RESETCONTENT, 0, 0);
									CB_SelectText(hWndPropEditValue, "", TRUE);
									break;
							};
							
							//Send a selection change message to the Value Combo
							SendMessage(hWnd, WM_COMMAND, GET_WM_COMMAND_MPS(IDC_PROPEDIT_VALUE, hWnd, LBN_SELCHANGE));
							break;
						}

						case IDC_PROPEDIT_VALUE:
						{	
							CHAR szBuffer[MAX_NAME_LEN+1];
							WCHAR wszBuffer[MAX_NAME_LEN+1];
							
							//Get new selection
							CB_GetSelectedText(hWndPropEditValue, szBuffer, MAX_NAME_LEN);
							
							//Verify Value
							XTEST(hWnd, VariantClear(&pPropInfo->vValues));
							ConvertToWCHAR(szBuffer, wszBuffer, MAX_NAME_LEN);
							if(FAILED(StringToVariant(wszBuffer, (DBTYPE)pPropInfo->vtType, &pPropInfo->vValues, CONV_VARBOOL)))
							{
								wMessageBox(hWnd, MB_TASKMODAL | MB_ICONERROR | MB_OK | MB_DEFBUTTON1, 
									wsz_ERROR, L"Unable to create Variant for type=%S, value=\"%S\"", GetVariantTypeName(pPropInfo->vtType), szBuffer);
								SetFocus(hWndPropEditValue);
 								return TRUE; //Don't Allow the Change
							}
							VariantToString(&pPropInfo->vValues, szBuffer, MAX_NAME_LEN, CONV_VARBOOL);
							LV_SetItemText(hWndPropNames, iSelRow, PROP_VALUE, szBuffer);
							break;
						}
					}
					break;
				}
			
				return FALSE;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_CLEARALL:
				{
					//Get the "this" pointer
					Busy();
					CPropDlg* pThis = (CPropDlg*)GetThis(hWnd);
					CListBox* pCListBox = pThis->m_pCListBox;
					HRESULT hr = S_OK;

					HWND hWndProp  = GetDlgItem(hWnd, IDL_PROPERTY);

					//Send a selection change message to the Value Combo
					SendMessage(hWnd, WM_COMMAND, GET_WM_COMMAND_MPS(IDC_PROPEDIT_VALUE, hWnd, LBN_SELCHANGE));
					
					//Delete all previous properties for our set
					ULONG cPropSets = 0;
					DBPROPSET* rgPropSets = NULL;
					
					//Now that the user has selected to clear all properties
					//We need to loop through all the checked
					//properties and just uncheck them...
					LONG iItems = SendMessage(hWndProp, LVM_GETITEMCOUNT, 0, 0);
					for(LONG i=0; i<iItems; i++)
					{
						//Get the Image and Param
						LONG lParam = LV_GetItemParam(hWndProp, i, 0);
						BOOL bChecked = LV_GetItemState(hWndProp, i, LVIS_STATEIMAGEMASK) & INDEXTOSTATEIMAGEMASK(STATE_CHECKED);

						//Only interested in actual items, (not headers or spaces)
						if(lParam == PARAM_NONE)
							continue;

						//Only interested in checked items
						if(!bChecked)
							continue;

						//Uncheck the property state
						LV_SetItemState(hWndProp, i, 0,  INDEXTOSTATEIMAGEMASK(STATE_UNCHECKED),  LVIS_STATEIMAGEMASK);
					}

					return 0;
				}
				
				case IDCANCEL:
				{
					//Cleanup any memory allocated
					FreeProperties(&cPropInfoSets, &rgPropInfoSets);
					SAFE_FREE(pStringBuffer);
					EndDialog(hWnd, FALSE);
					return 0;
				}

				case IDOK:
				{
					//Get the "this" pointer
					Busy();
					CPropDlg* pThis = (CPropDlg*)GetThis(hWnd);
					CListBox* pCListBox = pThis->m_pCListBox;
					HRESULT hr = S_OK;

					HWND hWndProp  = GetDlgItem(hWnd, IDL_PROPERTY);

					//Send a selection change message to the Value Combo
					SendMessage(hWnd, WM_COMMAND, GET_WM_COMMAND_MPS(IDC_PROPEDIT_VALUE, hWnd, LBN_SELCHANGE));
					
					//Delete all previous properties for our set
					ULONG cPropSets = 0;
					DBPROPSET* rgPropSets = NULL;
					
					//Now that the user has selected all the properties
					//they wish to have set, we need to loop through all the checked
					//properties and add them to our list...
					LONG iItems = SendMessage(hWndProp, LVM_GETITEMCOUNT, 0, 0);
					for(LONG i=0; i<iItems; i++)
					{
						//Get the Image and Param
						LONG lParam = LV_GetItemParam(hWndProp, i, 0);
						BOOL bChecked = LV_GetItemState(hWndProp, i, LVIS_STATEIMAGEMASK) & INDEXTOSTATEIMAGEMASK(STATE_CHECKED);

						//Only interested in actual items, (not headers or spaces)
						if(lParam == PARAM_NONE)
							continue;

						//Only interested in checked items
						if(!bChecked)
							continue;

						//Obtain which property this list item refers to...						//Now we have a real checked item so set it
						ULONG iPropInfoSet = LOWORD(lParam);
						ULONG iPropInfo = HIWORD(lParam);
						ASSERT(iPropInfoSet < cPropInfoSets);
						ASSERT(rgPropInfoSets && iPropInfo < rgPropInfoSets[iPropInfoSet].cPropertyInfos);
						ASSERT(rgPropInfoSets[iPropInfoSet].rgPropertyInfos);
						DBPROPINFOSET* pPropInfoSet = &rgPropInfoSets[iPropInfoSet];
						DBPROPINFO* pPropInfo = &pPropInfoSet->rgPropertyInfos[iPropInfo];

						//Now we have a real checked item so add it to our propset
						SetProperty(pPropInfo->dwPropertyID, pPropInfoSet->guidPropertySet, &cPropSets, &rgPropSets, DBTYPE_VARIANT, &pPropInfo->vValues, pPropInfo->dwFlags);
					}

					//Either just save the properties back in the propset provided
					if(pThis->m_pcPropSets && pThis->m_prgPropSets)
					{
						FreeProperties(&(*(pThis->m_pcPropSets)), &(*(pThis->m_prgPropSets)));
						*(pThis->m_pcPropSets) = cPropSets;
						*(pThis->m_prgPropSets) = rgPropSets;
					}
					//Or actually set the properties...
					else
					{
						//Now actually set the Properties
						switch(pThis->m_eSource)
						{
							case DATASOURCEINIT:
							case DATASOURCE:
							{	
								pCListBox->OutputPreMethod("IDBProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
								XTEST_(hWnd, hr = ((IDBProperties*)pThis->m_pIUnkProperties)->SetProperties(cPropSets, rgPropSets),S_OK);
								pCListBox->OutputPostMethod(hr, "IDBProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
								break;
							}
							case DATASOURCEADMIN:
							{	
								pCListBox->OutputPreMethod("IDBDataSourceAdmin::ModifyDataSource(%d, 0x%08x)", cPropSets, rgPropSets);
								XTEST_(hWnd, hr = ((IDBDataSourceAdmin*)pThis->m_pIUnkProperties)->ModifyDataSource(cPropSets, rgPropSets),S_OK);
								pCListBox->OutputPostMethod(hr, "IDBDataSourceAdmin::ModifyDataSource(%d, 0x%08x)", cPropSets, rgPropSets);
								break;
							}
							case SESSION:
							{	
								pCListBox->OutputPreMethod("ISessionProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
								XTEST_(hWnd, hr = ((ISessionProperties*)pThis->m_pIUnkProperties)->SetProperties(cPropSets, rgPropSets),S_OK);
								pCListBox->OutputPostMethod(hr, "ISessionProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
								break;
							}
							case COMMAND:
							{	
								pCListBox->OutputPreMethod("ICommandProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
								XTEST_(hWnd, hr = ((ICommandProperties*)pThis->m_pIUnkProperties)->SetProperties(cPropSets, rgPropSets),S_OK);
								pCListBox->OutputPostMethod(hr, "ICommandProperties::SetProperties(%d, 0x%08x)", cPropSets, rgPropSets);
								break;
							}

							default:
								ASSERT(!"Unhandled Property Source");
								break;
						};
					}

					//Cleanup
					Busy(OFF);
					if(hr == DB_S_ERRORSOCCURRED || hr == DB_E_ERRORSOCCURRED)
						DisplayPropErrors(hWnd, cPropSets, rgPropSets);
					if(pThis->m_pcPropSets==NULL || pThis->m_prgPropSets==NULL)
						FreeProperties(&cPropSets, &rgPropSets);
					if(FAILED(hr))
						return FALSE;

					FreeProperties(&cPropInfoSets, &rgPropInfoSets);
					SAFE_FREE(pStringBuffer);
					EndDialog(hWnd, TRUE);
					return 0;
				}
			}
			break;
		}//WM_COMMAND

		case WM_NOTIFY:
		{
			LV_DISPINFO* pDispInfo = (LV_DISPINFO*)lParam;
			NM_LISTVIEW* pListView = (NM_LISTVIEW*)lParam;
			
			//ListView
			switch(pDispInfo->hdr.code)
			{
				//Since we have "TwoClickActive" on this will get sent
				//Whenever a row is clicked on twice!
				//This functionality used to be done with NM_DBCLK
				case LVN_ITEMACTIVATE:
				{
					return FALSE;
				}

				case LVN_ITEMCHANGED:
				{
					//Check boxes
					if(pListView->uNewState & LVIS_STATEIMAGEMASK)
					{
						//Get the Current Param
						HWND hWndPropNames   = GetDlgItem(hWnd, IDL_PROPERTY);
						LONG lParam = LV_GetItemParam(hWndPropNames, pListView->iItem, pListView->iSubItem);

						//All lParam are PARAM_NONE for headers and non properties...
						//We don't want to change icons for those...
						if(lParam == PARAM_NONE && 
							(pListView->uNewState & INDEXTOSTATEIMAGEMASK(STATE_CHECKED) ||	pListView->uNewState & INDEXTOSTATEIMAGEMASK(STATE_UNCHECKED)))
						{
							LV_SetItemState(hWndPropNames, pListView->iItem, pListView->iSubItem,  0,  LVIS_STATEIMAGEMASK);
							return TRUE;
						}
					}
					return FALSE;
				}	

				case LVN_ITEMCHANGING:
				{
					//We Selected a "new" item
					if(pListView->uNewState & LVNI_FOCUSED &&
				 		pListView->uNewState & LVNI_SELECTED)
					{								
						CHAR szBuffer[MAX_NAME_LEN+1];
						HWND hWndPropNames   = GetDlgItem(hWnd, IDL_PROPERTY);
						
						//Display property value and option in the edit section
						HWND hWndPropEditName	= GetDlgItem(hWnd, IDE_PROPEDIT_NAME);
						HWND hWndPropEditType	= GetDlgItem(hWnd, IDC_PROPEDIT_TYPE);
						HWND hWndPropEditValue	= GetDlgItem(hWnd, IDC_PROPEDIT_VALUE);
						HWND hWndPropEditOptions= GetDlgItem(hWnd, IDC_PROPEDIT_OPTIONS);

						//Send a selection change message to the Value Combo
						SendMessage(hWnd, WM_COMMAND, GET_WM_COMMAND_MPS(IDC_PROPEDIT_VALUE, hWnd, LBN_SELCHANGE));
						
						//Clear Previous info
						SendMessage(hWndPropEditName,	WM_SETTEXT, 0, (LPARAM)"");
						SendMessage(hWndPropEditValue,	CB_RESETCONTENT,	0, 0);
						
						//Non-properties and headers have PARAM_NONE param
						if(pListView->lParam == PARAM_NONE)
						{
							//Disable Combos
							EnableWindow(hWndPropEditName, FALSE);
							EnableWindow(hWndPropEditType, FALSE);
							EnableWindow(hWndPropEditValue, FALSE);
							EnableWindow(hWndPropEditOptions, FALSE);
							return FALSE;  //Allow the change
						}

						LONG iItem = pListView->iItem;
						
						//Obtain PropInfo element
						ULONG iPropInfoSet = LOWORD(pListView->lParam);
						ULONG iPropInfo = HIWORD(pListView->lParam);
						ASSERT(iPropInfoSet < cPropInfoSets);
						ASSERT(rgPropInfoSets && iPropInfo < rgPropInfoSets[iPropInfoSet].cPropertyInfos);
						ASSERT(rgPropInfoSets[iPropInfoSet].rgPropertyInfos);
						DBPROPINFO* pPropInfo = &rgPropInfoSets[iPropInfoSet].rgPropertyInfos[iPropInfo];

						//Enable Combos
						EnableWindow(hWndPropEditName, TRUE);
						EnableWindow(hWndPropEditType, TRUE);
						EnableWindow(hWndPropEditValue, TRUE);
						EnableWindow(hWndPropEditOptions, TRUE);

						//Insert Name
						szBuffer[0] = EOL;
						LV_GetItemText(hWndPropNames, iItem, PROP_NAME, szBuffer, MAX_NAME_LEN);
						SendMessage(hWndPropEditName, WM_SETTEXT, 0, (LPARAM)szBuffer);
						
						//Set Current Type Selection
						CB_SelectItemValue(hWndPropEditType, pPropInfo->vtType);

						//InsertValue
						if(pPropInfo->vtType == DBTYPE_BOOL)
						{
							SendMessage(hWndPropEditValue, CB_ADDSTRING,	0, (LPARAM)"VARIANT_TRUE");
							SendMessage(hWndPropEditValue, CB_ADDSTRING,	0, (LPARAM)"VARIANT_FALSE");
						}
						szBuffer[0] = EOL;
						LV_GetItemText(hWndPropNames, iItem, PROP_VALUE, szBuffer, MAX_NAME_LEN);
						LONG iIndex = CB_SelectText(hWndPropEditValue, szBuffer, TRUE);

						//Now select the item
						SendMessage(hWndPropEditValue, CB_SETCURSEL,	iIndex, (LPARAM)0);

						//InsertOptions
						SendMessage(hWndPropEditOptions, CB_SETITEMDATA,0, (LPARAM)iItem);
						CB_SelectText(hWndPropEditOptions, pPropInfo->dwFlags == DBPROPOPTIONS_REQUIRED ? "REQUIRED" : "OPTIONAL", FALSE);
						return FALSE; //Allow the change
					}
				}
			}


		}//WM_NOTIFY
	}//switch message
		
	return FALSE;
}



////////////////////////////////////////////////////////////////
// CPropDlg::DisplayGetPropertyInfo
//
/////////////////////////////////////////////////////////////////
HRESULT CPropDlg::DisplayGetPropertyInfo(HWND hWnd, GUID guidPropertySet)
{
	CListBox* pCListBox = m_pCListBox;
	HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
	HRESULT hr = S_OK;
	CHAR szBuffer[MAX_NAME_LEN+1];

	DBPROPIDSET PropertyIDSet[1];
	PropertyIDSet[0].cPropertyIDs = 0;
	PropertyIDSet[0].rgPropertyIDs = NULL;
	PropertyIDSet[0].guidPropertySet = guidPropertySet;
	ULONG cPropertyIDSets = (guidPropertySet == IID_NULL ? 0 : 1);
	DBPROPIDSET* rgPropertyIDSets = cPropertyIDSets==0 ? NULL : PropertyIDSet;

	ULONG i,j,iItems = 0;
	ULONG cPropInfoSets = 0;
	DBPROPINFOSET* rgPropInfoSets = NULL;
	WCHAR* pStringBuffer = NULL;

	//Now actually get the Property Info
	switch(m_eSource)
	{
		case DATASOURCE:
		{	
			TESTC(pCListBox->OutputPreMethod("IDBProperties::GetPropertyInfo(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			XTEST(hWnd, hr = ((IDBProperties*)m_pIUnkPropInfo)->GetPropertyInfo(cPropertyIDSets, rgPropertyIDSets, &cPropInfoSets, &rgPropInfoSets, &pStringBuffer));
			TESTC(pCListBox->OutputPostMethod(hr, "IDBProperties::GetPropertyInfo(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			break;
		}
		case DATASOURCEADMIN:
		{	
			TESTC(pCListBox->OutputPreMethod("IDBDataSourceAdmin::GetCreateProperties(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			XTEST(hWnd, hr = ((IDBDataSourceAdmin*)m_pIUnkPropInfo)->GetCreationProperties(cPropertyIDSets, rgPropertyIDSets, &cPropInfoSets, &rgPropInfoSets, &pStringBuffer));
			TESTC(pCListBox->OutputPostMethod(hr, "IDBDataSourceAdmin::GetCreateProperties(%d, 0x%08x, &%d, &0x%08x, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropInfoSets, rgPropInfoSets, pStringBuffer));
			break;
		}
		default:
			ASSERT(!"Unhandled Property Source");
			break;
	};

	//Display PropertyNames in LeftPane, and Values in the right pane
	iItems = 0;
	for(i=0; i<cPropInfoSets; i++)
	{
		//We need to display this PropertySet
		CHAR* pszName = GetPropSetName(rgPropInfoSets[i].guidPropertySet);
		if(pszName == NULL)
		{
			WCHAR wszBuffer[MAX_NAME_LEN+1];
			StringFromGUID2(rgPropInfoSets[i].guidPropertySet, wszBuffer, MAX_NAME_LEN);
			ConvertToMBCS(wszBuffer, szBuffer, MAX_NAME_LEN);
			pszName = szBuffer;
		}

		LV_InsertItem(hWndPropNames, iItems, PROPINFO_NAME, pszName, PARAM_NONE, IMAGE_NORMAL);
		iItems++;

		//Now display all properties in the set
		for(j=0; j<rgPropInfoSets[i].cPropertyInfos; j++)
		{
			DBPROPINFO* pPropInfo = &rgPropInfoSets[i].rgPropertyInfos[j];
			
			//Determine which icon to display
			LONG iImage = pPropInfo->dwFlags & DBPROPFLAGS_WRITE ? IMAGE_NORMAL : IMAGE_READONLY;

			//PROPINFO_NAME (Indent from PropertySet)
			pszName = GetPropertyName(pPropInfo->dwPropertyID, rgPropInfoSets[i].guidPropertySet);
			if(pszName)
				sprintf(szBuffer, "     %s", pszName);
			else
				sprintf(szBuffer, "     0x%x", pPropInfo->dwPropertyID);
			LV_InsertItem(hWndPropNames, iItems, PROPINFO_NAME, szBuffer, 0, iImage);
			
			//PROPINFO_TYPE
			LV_InsertItem(hWndPropNames, iItems, PROPINFO_TYPE, GetVariantTypeName(pPropInfo->vtType));

			//PROPINFO_VALUE
			LV_InsertItem(hWndPropNames, iItems, PROPINFO_VALUE, GetVariantTypeName(pPropInfo->vValues.vt));

			//PROPINFO_FALGS
			sprintf(szBuffer, "0x%08x", pPropInfo->dwFlags);
			LV_InsertItem(hWndPropNames, iItems, PROPINFO_FLAGS, szBuffer);

			//PROPINFO_DESC
			TESTC(ConvertToMBCS(pPropInfo->pwszDescription, szBuffer, MAX_NAME_LEN));
			LV_InsertItem(hWndPropNames, iItems, PROPINFO_DESC, szBuffer);
			iItems++;
		}

		//Now display a space between sets...
		LV_InsertItem(hWndPropNames, iItems, 0, "", PARAM_NONE, IMAGE_NORMAL);
		iItems++;
	}

	//AutoSize Columns
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROPINFO_NAME,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROPINFO_TYPE,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROPINFO_VALUE,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROPINFO_FLAGS,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROPINFO_DESC,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	
CLEANUP:
	//Free PropertyInfo
	FreeProperties(&cPropInfoSets, &rgPropInfoSets);
	SAFE_FREE(pStringBuffer);
	return hr;
}
							

////////////////////////////////////////////////////////////////
// CPropDlg::GetPropertyInfoProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CPropDlg::GetPropertyInfoProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static GUID   guidPropertySet = IID_NULL;

	switch(message)
	{
		case WM_INITDIALOG:	
		{	
			EXC_BEGIN

			Busy();
			HRESULT hr = E_FAIL;

			//Save the "this" pointer
			CPropDlg* pThis = (CPropDlg*)SetThis(hWnd, (LPARAM)lParam);
			
			//Use Extended ListView Styles!
			HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
			HWND hWndPropSet	= GetDlgItem(hWnd, IDC_PROPSET);
			SendMessage(hWndPropNames, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES);

			//Window Title
			switch(pThis->m_eSource)
			{
				case DATASOURCE:
				{	
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"IDBProperties::GetPropertyInfo");
					break;
				}
				case DATASOURCEADMIN:
				{	
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"IDBDataSourceAdmin::GetCreationProperties");
					break;
				}
				default:
					ASSERT(!"Unhandled Property Source");
					break;
			};

			//ColumnHeaders
			LV_InsertColumn(hWndPropNames,   PROPINFO_NAME, "Property");
			LV_InsertColumn(hWndPropNames,   PROPINFO_TYPE, "Type");
			LV_InsertColumn(hWndPropNames,   PROPINFO_VALUE,"Value");
			LV_InsertColumn(hWndPropNames,   PROPINFO_FLAGS,"Flags");
			LV_InsertColumn(hWndPropNames,   PROPINFO_DESC, "Description");
			
			//Setup ImageList
			HIMAGELIST hPropImageList = ImageList_Create(16, 16, ILC_MASK, 1, 0 );
			
			//IDI_NORMAL
			HICON hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_NORMAL));
			ImageList_AddIcon(hPropImageList, hIcon);
			//IDI_READONLY - readonly checked icon
			hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_READONLY));
			ImageList_AddIcon(hPropImageList, hIcon);

			//Set image list to the ListView
			ListView_SetImageList(hWndPropNames, hPropImageList, LVSIL_SMALL);
//			ImageList_Destroy(hPropImageList);

			//Fill in the PROPSET Combo
			for(ULONG i=0; i<g_cPropSetMaps; i++)
			{
				LONG iSel = SendMessage(hWndPropSet, CB_ADDSTRING, 0, (LPARAM)g_rgPropSetMap[i].pszName);
				SendMessage(hWndPropSet, CB_SETITEMDATA, iSel, (LPARAM)i);
			
				if(guidPropertySet == *(g_rgPropSetMap[i].pGuid))
					SendMessage(hWndPropSet, CB_SETCURSEL, iSel, 0);
			}

			//Add DBPROPSET All (0, NULL)...
			LONG iSel = SendMessage(hWndPropSet, CB_ADDSTRING, 0, (LPARAM)"All Properties");
			SendMessage(hWndPropSet, CB_SETITEMDATA, iSel, (LPARAM)g_cPropSetMaps);

			//Default Selection
			iSel = SendMessage(hWndPropSet, CB_GETCURSEL, 0, 0);
			SendMessage(hWndPropSet, CB_SETCURSEL, iSel==CB_ERR || iSel>(LONG)g_cPropSetMaps ? g_cPropSetMaps : iSel, 0);

			//Display GetPropertyInfo
			TESTC(hr = pThis->DisplayGetPropertyInfo(hWnd, guidPropertySet));
			
		CLEANUP:
			CenterDialog(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}
		
		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//LBN_SELCHANGE ListBox Selection change
			switch(GET_WM_COMMAND_CMD(wParam, lParam))
			{
				//Selection change in a list box occurred
				case LBN_SELCHANGE:
				{	
					//Otherwise see which combo box has changed
					switch(GET_WM_COMMAND_ID(wParam, lParam))
					{
						case IDC_PROPSET:
						{	
							//Get the "this" pointer
							CPropDlg* pThis = (CPropDlg*)GetThis(hWnd);
							HWND hWndPropSet = GetDlgItem(hWnd, IDC_PROPSET);
							HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
							HRESULT hr = S_OK;
							Busy();

							//Get new selection
							LONG iSel = SendMessage(hWndPropSet, CB_GETCURSEL, 0, 0L);
							LONG iIndex = SendMessage(hWndPropSet, CB_GETITEMDATA, iSel, 0);
							
							guidPropertySet = IID_NULL;
							if(iIndex != CB_ERR && iIndex <(LONG)g_cPropSetMaps)
								guidPropertySet = *(g_rgPropSetMap[iIndex].pGuid); 
							
							//Now remove any previous items
							SendMessage(hWndPropNames, LVM_DELETEALLITEMS, 0, 0);

							//Display GetPropertyInfo
							hr = pThis->DisplayGetPropertyInfo(hWnd, guidPropertySet);

							Busy(OFF);
							return 0;
						}
					}
					return 0;
				}
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDOK:
				{
					EndDialog(hWnd, TRUE);
					return 0;
				}

				case IDCANCEL:
				{
					EndDialog(hWnd, FALSE);
					return 0;
				}
			}
			break;
		}//WM_COMMAND

	}//switch message
		
	return FALSE;
}



////////////////////////////////////////////////////////////////
// CPropDlg::DisplayGetProperties
//
/////////////////////////////////////////////////////////////////
HRESULT CPropDlg::DisplayGetProperties(HWND hWnd, GUID guidPropertySet)
{
	CListBox* pCListBox = m_pCListBox;
	HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
	HRESULT hr = S_OK;
	CHAR szBuffer[MAX_NAME_LEN+1];
	WCHAR wszBuffer[MAX_NAME_LEN+1];

	DBPROPIDSET PropertyIDSet[1];
	PropertyIDSet[0].cPropertyIDs = 0;
	PropertyIDSet[0].rgPropertyIDs = NULL;
	PropertyIDSet[0].guidPropertySet = guidPropertySet;
	ULONG cPropertyIDSets = (guidPropertySet == IID_NULL ? 0 : 1);
	DBPROPIDSET* rgPropertyIDSets = cPropertyIDSets==0 ? NULL : PropertyIDSet;

	ULONG i,j,iItems = 0;
	ULONG cPropSets = 0;
	DBPROPSET* rgPropSets = NULL;

	//Now actually get the Properties
	switch(m_eSource)
	{
		case DATASOURCE:
		{	
			TESTC(pCListBox->OutputPreMethod("IDBProperties::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			XTEST(hWnd, hr = ((IDBProperties*)m_pIUnkProperties)->GetProperties(cPropertyIDSets, rgPropertyIDSets, &cPropSets, &rgPropSets));
			TESTC(pCListBox->OutputPostMethod(hr, "IDBProperties::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			break;
		}
		case SESSION:
		{	
			TESTC(pCListBox->OutputPreMethod("ISessionProperties::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			XTEST(hWnd, hr = ((ISessionProperties*)m_pIUnkProperties)->GetProperties(cPropertyIDSets, rgPropertyIDSets, &cPropSets, &rgPropSets));
			TESTC(pCListBox->OutputPostMethod(hr, "ISessionProperties::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			break;
		}
		case COMMAND:
		{	
			TESTC(pCListBox->OutputPreMethod("ICommandProperties::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			XTEST(hWnd, hr = ((ICommandProperties*)m_pIUnkProperties)->GetProperties(cPropertyIDSets, rgPropertyIDSets, &cPropSets, &rgPropSets));
			TESTC(pCListBox->OutputPostMethod(hr, "ICommandProperties::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			break;
		}
		case ROWSET:
		{	
			TESTC(pCListBox->OutputPreMethod("IRowsetInfo::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			XTEST(hWnd, hr = ((IRowsetInfo*)m_pIUnkProperties)->GetProperties(cPropertyIDSets, rgPropertyIDSets, &cPropSets, &rgPropSets));
			TESTC(pCListBox->OutputPostMethod(hr, "IRowsetInfo::GetProperties(%d, 0x%08x, &%d, &0x%08x)", cPropertyIDSets, rgPropertyIDSets, cPropSets, rgPropSets));
			break;
		}

		default:
			ASSERT(!"Unhandled Property Source");
			break;
	};

	//Display PropertyNames in LeftPane, and Values in the right pane
	iItems = 0;
	for(i=0; i<cPropSets; i++)
	{
		//We need to display this PropertySet
		CHAR* pszName = GetPropSetName(rgPropSets[i].guidPropertySet);
		if(pszName == NULL)
		{
			WCHAR wszBuffer[MAX_NAME_LEN+1];
			StringFromGUID2(rgPropSets[i].guidPropertySet, wszBuffer, MAX_NAME_LEN);
			ConvertToMBCS(wszBuffer, szBuffer, MAX_NAME_LEN);
			pszName = szBuffer;
		}

		LV_InsertItem(hWndPropNames, iItems, PROP_NAME, pszName, PARAM_NONE, IMAGE_NORMAL);
		iItems++;

		//Now display all properties in the set
		for(j=0; j<rgPropSets[i].cProperties; j++)
		{
			DBPROP* pProp = &rgPropSets[i].rgProperties[j];
			
			//Determine which icon to display
			LONG iImage = IMAGE_NORMAL;

			//PROP_NAME (Indent from PropertySet)
			pszName = GetPropertyName(pProp->dwPropertyID, rgPropSets[i].guidPropertySet);
			if(pszName)
				sprintf(szBuffer, "     %s", pszName);
			else
				sprintf(szBuffer, "     0x%x", pProp->dwPropertyID);
			LV_InsertItem(hWndPropNames, iItems, PROP_NAME, szBuffer, 0, iImage);
			
			//PROP_TYPE
			LV_InsertItem(hWndPropNames, iItems, PROP_TYPE, GetVariantTypeName(V_VT(&pProp->vValue)));

			//PROP_VALUE
			VariantToString(&pProp->vValue, szBuffer, MAX_NAME_LEN, CONV_VARBOOL);
			LV_InsertItem(hWndPropNames, iItems, PROP_VALUE, szBuffer);

			//PROP_OPTIONS
			if(pProp->dwOptions == DBPROPOPTIONS_OPTIONAL)
				LV_InsertItem(hWndPropNames, iItems, PROP_OPTIONS, "OPTIONAL");
			else
				LV_InsertItem(hWndPropNames, iItems, PROP_OPTIONS, "REQUIRED");

			//PROP_COLID
			DBIDToString(&pProp->colid, wszBuffer, MAX_NAME_LEN);
			ConvertToMBCS(wszBuffer, szBuffer, MAX_NAME_LEN);
			LV_InsertItem(hWndPropNames, iItems, PROP_COLID, szBuffer);
			iItems++;
		}

		//Now display a space between sets...
		LV_InsertItem(hWndPropNames, iItems, 0, "", PARAM_NONE, IMAGE_NORMAL);
		iItems++;
	}

	//AutoSize Columns
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_NAME,		(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_TYPE,		(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_VALUE,		(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_OPTIONS,	(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	SendMessage(hWndPropNames, LVM_SETCOLUMNWIDTH, (WPARAM)PROP_COLID,		(LPARAM)LVSCW_AUTOSIZE_USEHEADER);
	
CLEANUP:
	FreeProperties(&cPropSets, &rgPropSets);
	return hr;
}


////////////////////////////////////////////////////////////////
// CPropDlg::GetPropertiesProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CPropDlg::GetPropertiesProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static GUID   guidPropertySet = IID_NULL;

	switch(message)
	{
		case WM_INITDIALOG:	
		{	
			EXC_BEGIN

			Busy();
			HRESULT hr = S_OK;

			//Save the "this" pointer
			CPropDlg* pThis = (CPropDlg*)SetThis(hWnd, (LPARAM)lParam);
			
			//Use Extended ListView Styles!
			HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
			HWND hWndPropSet	= GetDlgItem(hWnd, IDC_PROPSET);
			SendMessage(hWndPropNames, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES, LVS_EX_FULLROWSELECT | LVS_EX_GRIDLINES | LVS_EX_TWOCLICKACTIVATE | LVS_EX_SUBITEMIMAGES);

			//Now actually get the Properties
			switch(pThis->m_eSource)
			{
				case DATASOURCE:
				{	
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"IDBProperties::GetProperties");
					break;
				}
				case SESSION:
				{	
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"ISessionProperties::GetProperties");
					break;
				}
				case COMMAND:
				{	
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"ICommandProperties::GetProperties");
					break;
				}
				case ROWSET:
				{	
					SendMessage(hWnd, WM_SETTEXT, 0, (LPARAM)"IRowsetInfo::GetProperties");
					break;
				}

				default:
					ASSERT(!"Unhandled Property Source");
					break;
			};

			//ColumnHeaders
			LV_InsertColumn(hWndPropNames,   PROP_NAME,		"Property");
			LV_InsertColumn(hWndPropNames,   PROP_TYPE,		"Type");
			LV_InsertColumn(hWndPropNames,   PROP_VALUE,	"Value");
			LV_InsertColumn(hWndPropNames,   PROP_OPTIONS,	"Options");
			LV_InsertColumn(hWndPropNames,   PROP_COLID,	"ColID");
			
			//Setup ImageList
			HIMAGELIST hPropImageList = ImageList_Create(16, 16, ILC_MASK, 1, 0 );
			
			//IDI_NORMAL
			HICON hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_NORMAL));
			ImageList_AddIcon(hPropImageList, hIcon);
			//IDI_READONLY - readonly checked icon
			hIcon = LoadIcon(pThis->m_hInst, MAKEINTRESOURCE(IDI_READONLY));
			ImageList_AddIcon(hPropImageList, hIcon);

			//Set image list to the ListView
			ListView_SetImageList(hWndPropNames, hPropImageList, LVSIL_SMALL);
//			ImageList_Destroy(hPropImageList);

			//Fill in the PROPSET Combo
			for(ULONG i=0; i<g_cPropSetMaps; i++)
			{
				LONG iSel = SendMessage(hWndPropSet, CB_ADDSTRING, 0, (LPARAM)g_rgPropSetMap[i].pszName);
				SendMessage(hWndPropSet, CB_SETITEMDATA, iSel, (LPARAM)i);
			
				if(guidPropertySet == *(g_rgPropSetMap[i].pGuid))
					SendMessage(hWndPropSet, CB_SETCURSEL, iSel, 0);
			}

			//Add DBPROPSET All (0, NULL)...
			LONG iSel = SendMessage(hWndPropSet, CB_ADDSTRING, 0, (LPARAM)"All Properties");
			SendMessage(hWndPropSet, CB_SETITEMDATA, iSel, (LPARAM)g_cPropSetMaps);

			//Default Selection
			iSel = SendMessage(hWndPropSet, CB_GETCURSEL, 0, 0);
			SendMessage(hWndPropSet, CB_SETCURSEL, iSel==CB_ERR || iSel>(LONG)g_cPropSetMaps ? g_cPropSetMaps : iSel, 0);

			//Display GetProperties
			hr = pThis->DisplayGetProperties(hWnd, guidPropertySet);
				
			CenterDialog(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}
		
		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//LBN_SELCHANGE ListBox Selection change
			switch(GET_WM_COMMAND_CMD(wParam, lParam))
			{
				//Selection change in a list box occurred
				case LBN_SELCHANGE:
				{	
					//Otherwise see which combo box has changed
					switch(GET_WM_COMMAND_ID(wParam, lParam))
					{
						case IDC_PROPSET:
						{	
							//Get the "this" pointer
							CPropDlg* pThis = (CPropDlg*)GetThis(hWnd);
							HWND hWndPropSet = GetDlgItem(hWnd, IDC_PROPSET);
							HWND hWndPropNames  = GetDlgItem(hWnd, IDL_PROPERTY);
							HRESULT hr = S_OK;
							Busy();

							//Get new selection
							LONG iSel = SendMessage(hWndPropSet, CB_GETCURSEL, 0, 0L);
							LONG iIndex = SendMessage(hWndPropSet, CB_GETITEMDATA, iSel, 0);
							
							guidPropertySet = IID_NULL;
							if(iIndex != CB_ERR && iIndex <(LONG)g_cPropSetMaps)
								guidPropertySet = *(g_rgPropSetMap[iIndex].pGuid); 
							
							//Now remove any previous items
							SendMessage(hWndPropNames, LVM_DELETEALLITEMS, 0, 0);

							//Display GetProperties
							hr = pThis->DisplayGetProperties(hWnd, guidPropertySet);

							Busy(OFF);
							return 0;
						}
					}
					return 0;
				}
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDOK:
				{
					EndDialog(hWnd, TRUE);
					return 0;
				}

				case IDCANCEL:
				{
					EndDialog(hWnd, FALSE);
					return 0;
				}
			}
			break;
		}//WM_COMMAND

	}//switch message
		
	return FALSE;
}




////////////////////////////////////////////////////////////////
// COptionsDlg::COptionsDlg
//
/////////////////////////////////////////////////////////////////
COptionsDlg::COptionsDlg(HWND hWnd, HINSTANCE hInst, CMainWindow* pCMainWindow)
	: CDialogBase(hWnd, hInst)
{
	ASSERT(pCMainWindow);
	
	//Data
	m_pCMainWindow = pCMainWindow;

	//DataSource
	m_hWndDataSourceOptions	= NULL;
	m_dwDSOCLSCTX			= CLSCTX_INPROC_SERVER;
	m_dwDataSourceOpts		= CONNECT_INITIALIZE | CONNECT_SETPROPERTIES | CONNECT_CREATESESSION | CONNECT_CREATECOMMAND;

	//Rowset
	m_hWndRowsetOptions		= NULL;
	m_dwBindingType			= DBTYPE_WSTR;
	m_dwConvFlags			= CONV_ALPHABOOL;
	m_dwRowsetOpts			= ROWSET_AUTOQI | ROWSET_SETDEFAULTPROPS;

	//Command
	m_hWndCommandOptions	= NULL;
	m_guidDialect			= DBGUID_DEFAULT;
	wcscpy(m_wszDialect, L"DBGUID_DEFAULT");
	m_dwCommandOpts			= COMMAND_ROWSAFFECTED | COMMAND_RELEASEROWSET;

	//Notifcations
	m_hWndNotifyOptions		= NULL;
	m_dwNotifyOpts			= NOTIFY_METHODCALLS | NOTIFY_IDBASYNCHNOTIFY | NOTIFY_IROWSETNOTIFY | NOTIFY_IROWPOSITIONCHANGE;

	//Error
	m_hWndErrorOptions		= NULL;

	//Load Options from the Registry
	RestoreOptions();
}


////////////////////////////////////////////////////////////////
// COptionsDlg::~COptionsDlg
//
/////////////////////////////////////////////////////////////////
COptionsDlg::~COptionsDlg()
{
	m_hWnd = NULL;
}


////////////////////////////////////////////////////////////////
// COptionsDlg::SaveOptions
//
/////////////////////////////////////////////////////////////////
HRESULT COptionsDlg::SaveOptions()
{
	//DataSource
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "CLSCTX",					m_dwDSOCLSCTX);
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "ConnectOpts",			m_dwDataSourceOpts);

	//Rowset
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "BindingType",			m_dwBindingType);
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "ConvFlags",				m_dwConvFlags);
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "RowsetOpts",				m_dwRowsetOpts);

	//Command
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "DefaultDialec",			m_wszDialect);
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "CommandOpts",			m_dwCommandOpts);

	//Notifications
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "NotifyOpts",				m_dwNotifyOpts);

	//Errors
	SetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "ErrorPostingFlags",		g_dwErrorPost);
	return S_OK;
}


////////////////////////////////////////////////////////////////
// COptionsDlg::RestoreOptions
//
/////////////////////////////////////////////////////////////////
HRESULT COptionsDlg::RestoreOptions()
{
	//DataSource
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "CLSCTX",					&m_dwDSOCLSCTX);
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "ConnectOpts",			&m_dwDataSourceOpts);

	//Rowset
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "BindingType",			&m_dwBindingType);
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "ConvFlags",				&m_dwConvFlags);
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "RowsetOpts",				&m_dwRowsetOpts);

	//Command
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "DefaultDialec",			m_wszDialect, MAX_NAME_LEN);
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "CommandOpts",			&m_dwCommandOpts);

	//Notifications
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "NotifyOpts",				&m_dwNotifyOpts);

	//Errors
	GetRegEntry(HKEY_ROWSETVIEWER, szOPTIONS_KEY, "ErrorPostingFlags",		&g_dwErrorPost);
	return S_OK;
}


////////////////////////////////////////////////////////////////
// COptionsDlg::Display
//
/////////////////////////////////////////////////////////////////
ULONG COptionsDlg::Display()
{
    PROPSHEETPAGE psp[5];
    PROPSHEETHEADER psh;

    //Header
    psh.dwSize			= sizeof(PROPSHEETHEADER);
    psh.dwFlags			= PSH_PROPSHEETPAGE | PSH_NOAPPLYNOW;
    psh.hwndParent		= m_hWnd;
    psh.hInstance		= m_hInst;
    psh.pszIcon			= NULL;
    psh.pszCaption		= "Options";
    psh.nPages			= NUMELE(psp);
    psh.nStartPage		= 0;
    psh.ppsp			= (LPCPROPSHEETPAGE) &psp;

    //DataSource
	psp[0].dwSize		= sizeof(PROPSHEETPAGE);
    psp[0].dwFlags		= PSP_USETITLE;
    psp[0].hInstance	= m_hInst;
    psp[0].pszTemplate	= MAKEINTRESOURCE(IDD_FULLCONNECT_OPTIONS);
    psp[0].pszIcon		= NULL;
    psp[0].pfnDlgProc	= DataSourceOptionsProc;
    psp[0].pszTitle		= "DataSource";
    psp[0].lParam		= (LONG)this;

    //Command
	psp[1].dwSize		= sizeof(PROPSHEETPAGE);
    psp[1].dwFlags		= PSP_USETITLE;
    psp[1].hInstance	= m_hInst;
    psp[1].pszTemplate	= MAKEINTRESOURCE(IDD_OPTIONS_COMMAND);
    psp[1].pszIcon		= NULL;
    psp[1].pfnDlgProc	= CommandOptionsProc;
    psp[1].pszTitle		= "Command";
    psp[1].lParam		= (LONG)this;

	//Rowset
	psp[2].dwSize		= sizeof(PROPSHEETPAGE);
    psp[2].dwFlags		= PSP_USETITLE;
    psp[2].hInstance	= m_hInst;
    psp[2].pszTemplate	= MAKEINTRESOURCE(IDD_OPTIONS_ROWSET);
    psp[2].pszIcon		= NULL;
    psp[2].pfnDlgProc	= RowsetOptionsProc;
    psp[2].pszTitle		= "Rowset";
    psp[2].lParam		= (LONG)this;

    //Notifcations
	psp[3].dwSize		= sizeof(PROPSHEETPAGE);
    psp[3].dwFlags		= PSP_USETITLE;
    psp[3].hInstance	= m_hInst;
    psp[3].pszTemplate	= MAKEINTRESOURCE(IDD_OPTIONS_NOTIFICATIONS);
    psp[3].pszIcon		= NULL;
    psp[3].pfnDlgProc	= NotifyOptionsProc;
    psp[3].pszTitle		= "Notify";
    psp[3].lParam		= (LONG)this;

    //Errors
	psp[4].dwSize		= sizeof(PROPSHEETPAGE);
    psp[4].dwFlags		= PSP_USETITLE;
    psp[4].hInstance	= m_hInst;
    psp[4].pszTemplate	= MAKEINTRESOURCE(IDD_OPTIONS_ERRORS);
    psp[4].pszIcon		= NULL;
    psp[4].pfnDlgProc	= ErrorOptionsProc;
    psp[4].pszTitle		= "Errors";
    psp[4].lParam		= (LONG)this;

    return PropertySheet(&psh);
}


////////////////////////////////////////////////////////////////
// COptionsDlg::InitDataSourceOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::InitDataSourceOptions(HWND hWnd)
{
	m_hWndDataSourceOptions = hWnd;
	
	//Set CLSCTX to previous saved values
	CheckDlgButton(hWnd, IDB_INPROC_SERVER,			m_dwDSOCLSCTX & CLSCTX_INPROC_SERVER			? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_LOCAL_SERVER,			m_dwDSOCLSCTX & CLSCTX_LOCAL_SERVER			? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_REMOTE_SERVER,			m_dwDSOCLSCTX & CLSCTX_REMOTE_SERVER			? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INPROC_HANDLER,		m_dwDSOCLSCTX & CLSCTX_INPROC_HANDLER			? BST_CHECKED : BST_UNCHECKED);

	//Currently there is no way to pass RemoteServer in GetDataSource methods...
	EnableWindow(GetDlgItem(hWnd, IDE_REMOTESERVER),FALSE);

	//Set Connection Options to previous saved values
	CheckDlgButton(hWnd, IDB_INIT_INITIALIZE,		m_dwDataSourceOpts & CONNECT_INITIALIZE		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INIT_SETPROPERTIES,	m_dwDataSourceOpts & CONNECT_SETPROPERTIES		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INIT_CREATESESSION,	m_dwDataSourceOpts & CONNECT_CREATESESSION		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_INIT_CREATECOMMAND,	m_dwDataSourceOpts & CONNECT_CREATECOMMAND		? BST_CHECKED : BST_UNCHECKED);

	//Set Other Options
	CheckDlgButton(hWnd, IDB_USESERVICECOMP,		m_dwDataSourceOpts & CONNECT_USESERVICECOMP	? BST_CHECKED : BST_UNCHECKED);
	EnableWindow(GetDlgItem(hWnd, IDB_USESERVICECOMP), (BOOL)m_pCMainWindow->m_pCConnectDlg->pIDataInitialize());
	return TRUE;
}


////////////////////////////////////////////////////////////////
// COptionsDlg::GetDataSourceOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::GetDataSourceOptions()
{
	//Obtain CLSCTX
	m_dwDSOCLSCTX = 0;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_INPROC_SERVER))
		m_dwDSOCLSCTX |= CLSCTX_INPROC_SERVER;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_LOCAL_SERVER))
		m_dwDSOCLSCTX |= CLSCTX_LOCAL_SERVER;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_REMOTE_SERVER))
		m_dwDSOCLSCTX |= CLSCTX_REMOTE_SERVER;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_INPROC_HANDLER))
		m_dwDSOCLSCTX |= CLSCTX_INPROC_HANDLER;

	//Obtain Connection Options
	m_dwDataSourceOpts = 0;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_INIT_INITIALIZE))
		m_dwDataSourceOpts |= CONNECT_INITIALIZE;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_INIT_SETPROPERTIES))
		m_dwDataSourceOpts |= CONNECT_SETPROPERTIES;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_INIT_CREATESESSION))
		m_dwDataSourceOpts |= CONNECT_CREATESESSION;
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_INIT_CREATECOMMAND))
		m_dwDataSourceOpts |= CONNECT_CREATECOMMAND;

	//Obtain Other Options
	if(IsDlgButtonChecked(m_hWndDataSourceOptions, IDB_USESERVICECOMP))
		m_dwDataSourceOpts |= CONNECT_USESERVICECOMP;
	return TRUE;	
}


////////////////////////////////////////////////////////////////
// COptionsDlg::InitCommandOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::InitCommandOptions(HWND hWnd)
{
	m_hWndCommandOptions = hWnd;

	//Display RowAffected
	CheckDlgButton(hWnd, IDB_DISPLAY_ROWSAFFECTED, (m_dwCommandOpts & COMMAND_ROWSAFFECTED) ? BST_CHECKED : BST_UNCHECKED);

	//Display ReleaseRowset
	CheckDlgButton(hWnd, IDB_RELEASEROWSET, (m_dwCommandOpts & COMMAND_RELEASEROWSET) ? BST_CHECKED : BST_UNCHECKED);
	return TRUE;	
}


////////////////////////////////////////////////////////////////
// COptionsDlg::GetCommandOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::GetCommandOptions()
{
	HWND hWnd = m_hWndCommandOptions;

	//Display RowsAffected
	ENABLE_BIT(m_dwCommandOpts, COMMAND_ROWSAFFECTED, IsDlgButtonChecked(m_hWndCommandOptions, IDB_DISPLAY_ROWSAFFECTED));

	//Display ReleaseRowset
	ENABLE_BIT(m_dwCommandOpts, COMMAND_RELEASEROWSET, IsDlgButtonChecked(m_hWndCommandOptions, IDB_RELEASEROWSET));
	return TRUE;	
}


////////////////////////////////////////////////////////////////
// COptionsDlg::InitRowsetOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::InitRowsetOptions(HWND hWnd)
{
	m_hWndRowsetOptions = hWnd;
	
	//BindingType
	if(m_dwBindingType == DBTYPE_UDT)
		CheckDlgButton(hWnd, IDB_BINDINGTYPE_NATIVE, BST_CHECKED);
	else if(m_dwBindingType == DBTYPE_STR)
		CheckDlgButton(hWnd, IDB_BINDINGTYPE_STR, BST_CHECKED);
	else if(m_dwBindingType == DBTYPE_BYTES)
		CheckDlgButton(hWnd, IDB_BINDINGTYPE_BYTES, BST_CHECKED);
	else if(m_dwBindingType == DBTYPE_VARIANT)
		CheckDlgButton(hWnd, IDB_BINDINGTYPE_VARIANT, BST_CHECKED);
	else
	{
		m_dwBindingType = DBTYPE_WSTR;
		CheckDlgButton(hWnd, IDB_BINDINGTYPE_WSTR, BST_CHECKED);
	}
	
	//BLOB Binding Type
	if(m_dwRowsetOpts & ROWSET_BLOB_ISEQSTREAM)
		CheckDlgButton(hWnd, IDB_BLOB_ISEQSTREAM, BST_CHECKED);
	else if(m_dwRowsetOpts & ROWSET_BLOB_ILOCKBYTES)
		CheckDlgButton(hWnd, IDB_BLOB_ILOCKBYTES, BST_CHECKED);
	else if(m_dwRowsetOpts & ROWSET_BLOB_ISTORAGE)
		CheckDlgButton(hWnd, IDB_BLOB_ISTORAGE, BST_CHECKED);
	else if(m_dwRowsetOpts & ROWSET_BLOB_ISTREAM)
		CheckDlgButton(hWnd, IDB_BLOB_ISTREAM, BST_CHECKED);
	else 
		CheckDlgButton(hWnd, IDB_BLOB_INLINE, BST_CHECKED);

	//Bookmark Column
	CheckDlgButton(hWnd, IDB_BIND_BOOKMARK, (m_dwRowsetOpts & ROWSET_BINDBOOKMARK) ? BST_CHECKED : BST_UNCHECKED);

	//Hexidecimal
	CheckDlgButton(hWnd, IDB_DISPLAY_HEXADECIMAL, (m_dwConvFlags & CONV_HEX) ? BST_CHECKED : BST_UNCHECKED);

	//AutoQI
	CheckDlgButton(hWnd, IDB_AUTOQI, (m_dwRowsetOpts & ROWSET_AUTOQI) ? BST_CHECKED : BST_UNCHECKED);

	//DefaultProps
	CheckDlgButton(hWnd, IDB_SETDEFAULTPROPS, (m_dwRowsetOpts & ROWSET_SETDEFAULTPROPS) ? BST_CHECKED : BST_UNCHECKED);

	//GetAllRows
	if(m_dwRowsetOpts & ROWSET_GETALLROWS)
		CheckDlgButton(hWnd, IDB_GETALLROWS, BST_CHECKED);
	else if(m_dwRowsetOpts & ROWSET_GETNOROWS)
		CheckDlgButton(hWnd, IDB_GETNOROWS, BST_CHECKED);
	else 
		CheckDlgButton(hWnd, IDB_GETFEWROWS, BST_CHECKED);
	return TRUE;	
}


////////////////////////////////////////////////////////////////
// COptionsDlg::GetRowsetOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::GetRowsetOptions()
{
	//BindingType
	if(IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BINDINGTYPE_NATIVE))
		m_dwBindingType = DBTYPE_UDT;
	else if(IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BINDINGTYPE_STR))
		m_dwBindingType = DBTYPE_STR;
	else if(IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BINDINGTYPE_BYTES))
		m_dwBindingType = DBTYPE_BYTES;
	else if(IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BINDINGTYPE_VARIANT))
		m_dwBindingType = DBTYPE_VARIANT;
	else
		m_dwBindingType = DBTYPE_WSTR;

	//BLOB Binding Type
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_BLOB_ISEQSTREAM,	IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BLOB_ISEQSTREAM));
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_BLOB_ILOCKBYTES,	IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BLOB_ILOCKBYTES));
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_BLOB_ISTORAGE,	IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BLOB_ISTORAGE));
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_BLOB_ISTREAM,		IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BLOB_ISTREAM));

	//Hexidecimal
	ENABLE_BIT(m_dwConvFlags, CONV_HEX, IsDlgButtonChecked(m_hWndRowsetOptions, IDB_DISPLAY_HEXADECIMAL));

	//Bookmark Column
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_BINDBOOKMARK, IsDlgButtonChecked(m_hWndRowsetOptions, IDB_BIND_BOOKMARK));

	//AutoQI
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_AUTOQI, IsDlgButtonChecked(m_hWndRowsetOptions, IDB_AUTOQI));

	//DefaultProps
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_SETDEFAULTPROPS, IsDlgButtonChecked(m_hWndRowsetOptions, IDB_SETDEFAULTPROPS));

	//GetAllRows
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_GETALLROWS, IsDlgButtonChecked(m_hWndRowsetOptions, IDB_GETALLROWS));
	ENABLE_BIT(m_dwRowsetOpts, ROWSET_GETNOROWS, IsDlgButtonChecked(m_hWndRowsetOptions, IDB_GETNOROWS));
	return TRUE;	
}



////////////////////////////////////////////////////////////////
// COptionsDlg::InitNotifyOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::InitNotifyOptions(HWND hWnd)
{
	m_hWndNotifyOptions = hWnd;
	
	//Display
	CheckDlgButton(hWnd, IDB_DISPLAY_METHODCALLS,		(m_dwNotifyOpts & NOTIFY_METHODCALLS)	? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_DISPLAY_OTHERMETHODCALLS,	(m_dwNotifyOpts & NOTIFY_OTHERCALLS)	? BST_CHECKED : BST_UNCHECKED);
	
	//IUnknown
	CheckDlgButton(hWnd, IDB_DISPLAY_ADDREF,			(m_dwNotifyOpts & NOTIFY_ADDREF)		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_DISPLAY_RELEASE,			(m_dwNotifyOpts & NOTIFY_RELEASE)		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_DISPLAY_QUERYINTERFACE,	(m_dwNotifyOpts & NOTIFY_QUERYINTEFACE) ? BST_CHECKED : BST_UNCHECKED);

	//Notifications
	CheckDlgButton(hWnd, IDB_DISPLAY_IDBASYNCHNOTIFY,	(m_dwNotifyOpts & NOTIFY_IDBASYNCHNOTIFY)	? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_DISPLAY_IROWSETNOTIFY,		(m_dwNotifyOpts & NOTIFY_IROWSETNOTIFY)		? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_DISPLAY_IROWPOSITIONCHANGE,(m_dwNotifyOpts & NOTIFY_IROWPOSITIONCHANGE)? BST_CHECKED : BST_UNCHECKED);

	//Timings
	CheckDlgButton(hWnd, IDB_DISPLAY_TIMINGS,			(m_dwNotifyOpts & NOTIFY_TIMINGS)		? BST_CHECKED : BST_UNCHECKED);
	return TRUE;	
}


/////////////////////////////////////////////////////////////////
// COptionsDlg::GetNotifyOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::GetNotifyOptions()
{
	//Display
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_METHODCALLS,	IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_METHODCALLS));
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_OTHERCALLS,	IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_OTHERMETHODCALLS));

	//IUnknown
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_ADDREF,		IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_ADDREF));
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_RELEASE,		IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_RELEASE));
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_QUERYINTEFACE,IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_QUERYINTERFACE));

	//Notiifcations
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_IDBASYNCHNOTIFY,		IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_IDBASYNCHNOTIFY));
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_IROWSETNOTIFY,		IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_IROWSETNOTIFY));
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_IROWPOSITIONCHANGE,	IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_IROWPOSITIONCHANGE));

	//Timins
	ENABLE_BIT(m_dwNotifyOpts, NOTIFY_TIMINGS,		IsDlgButtonChecked(m_hWndNotifyOptions, IDB_DISPLAY_TIMINGS));
	return TRUE;	
}


////////////////////////////////////////////////////////////////
// COptionsDlg::InitErrorOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::InitErrorOptions(HWND hWnd)
{
	m_hWndErrorOptions = hWnd;
	
	//ErrorInfo
	if(GetErrorPosting(EP_ERRORINFO_ALWAYS))
		CheckDlgButton(hWnd, IDB_ERRORINFO_ALWAYS,		BST_CHECKED);
	if(GetErrorPosting(EP_ERRORINFO_FAILURE))
		CheckDlgButton(hWnd, IDB_ERRORINFO_FAILURE,		BST_CHECKED);
	if(GetErrorPosting(EP_ERRORINFO_NEVER))
		CheckDlgButton(hWnd, IDB_ERRORINFO_NEVER,		BST_CHECKED);

	//HRESULTs
	if(GetErrorPosting(EP_HRESULT_ALWAYS))
		CheckDlgButton(hWnd, IDB_HRESULT_ALWAYS,		BST_CHECKED);
	if(GetErrorPosting(EP_HRESULT_NOERRORINFO))
		CheckDlgButton(hWnd, IDB_HRESULT_NOERRORINFO,	BST_CHECKED);
	if(GetErrorPosting(EP_HRESULT_FAILURE))
		CheckDlgButton(hWnd, IDB_HRESULT_FAILURE,		BST_CHECKED);
	if(GetErrorPosting(EP_HRESULT_NEVER))
		CheckDlgButton(hWnd, IDB_HRESULT_NEVER,			BST_CHECKED);

	//RefCounts
	CheckDlgButton(hWnd, IDB_REFCOUNT_FAILURE,	GetErrorPosting(EP_REFCOUNT_FAILURE) ? BST_CHECKED : BST_UNCHECKED);
	return TRUE;	
}


/////////////////////////////////////////////////////////////////
// COptionsDlg::GetErrorOptions
//
/////////////////////////////////////////////////////////////////
BOOL COptionsDlg::GetErrorOptions()
{
	//ErrorInfo
	SetErrorPosting(EP_ERRORINFO_ALWAYS,	IsDlgButtonChecked(m_hWndErrorOptions, IDB_ERRORINFO_ALWAYS));
	SetErrorPosting(EP_ERRORINFO_FAILURE,	IsDlgButtonChecked(m_hWndErrorOptions, IDB_ERRORINFO_FAILURE));
	SetErrorPosting(EP_ERRORINFO_NEVER,		IsDlgButtonChecked(m_hWndErrorOptions, IDB_ERRORINFO_NEVER));

	//HRESULTs
	SetErrorPosting(EP_HRESULT_ALWAYS,		IsDlgButtonChecked(m_hWndErrorOptions, IDB_HRESULT_ALWAYS));
	SetErrorPosting(EP_HRESULT_NOERRORINFO,	IsDlgButtonChecked(m_hWndErrorOptions, IDB_HRESULT_NOERRORINFO));
	SetErrorPosting(EP_HRESULT_FAILURE,		IsDlgButtonChecked(m_hWndErrorOptions, IDB_HRESULT_FAILURE));
	SetErrorPosting(EP_HRESULT_NEVER,		IsDlgButtonChecked(m_hWndErrorOptions, IDB_HRESULT_NEVER));

	//HRESULTs
	SetErrorPosting(EP_REFCOUNT_FAILURE,	IsDlgButtonChecked(m_hWndErrorOptions, IDB_REFCOUNT_FAILURE));
	return TRUE;	
}


////////////////////////////////////////////////////////////////
// COptionsDlg::DataSourceOptionsProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK COptionsDlg::DataSourceOptionsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE * ps;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN

			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			COptionsDlg* pThis = (COptionsDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitDataSourceOptions(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_REMOTE_SERVER:
					//Currently always disabled since GetDataSource can't specify remote machine
//					EnableWindow(GetDlgItem(hWnd, IDE_REMOTESERVER), IsDlgButtonChecked(hWnd, IDB_REMOTE_SERVER));
					return 0;
			}
		
			return FALSE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					// initialize the controls
					pThis->m_hWndDataSourceOptions = hWnd;
					return 0;
				}

				case PSN_KILLACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					//Get DataSourceOptions
					if(!pThis->GetDataSourceOptions())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					//Save in the registry
					pThis->SaveOptions();
					return 0;
				}

				case PSN_RESET:
					// rest to the original values
					return 0;
			}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// COptionsDlg::CommandOptionsProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK COptionsDlg::CommandOptionsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE * ps;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN

			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			COptionsDlg* pThis = (COptionsDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitCommandOptions(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					// initialize the controls
					pThis->m_hWndCommandOptions = hWnd;
					return 0;
				}

				case PSN_KILLACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					//Get Command Options
					if(!pThis->GetCommandOptions())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:
					return 0;

				case PSN_RESET:
					// rest to the original values
					return 0;
			}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// COptionsDlg::RowsetOptionsProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK COptionsDlg::RowsetOptionsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE * ps;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN

			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			COptionsDlg* pThis = (COptionsDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitRowsetOptions(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					// initialize the controls
					pThis->m_hWndRowsetOptions = hWnd;
					return 0;
				}

				case PSN_KILLACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					//Get RowsetOptions
					if(!pThis->GetRowsetOptions())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:
					return 0;

				case PSN_RESET:
					// rest to the original values
					return 0;
			}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// COptionsDlg::NotifyOptionsProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK COptionsDlg::NotifyOptionsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE * ps;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN

			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			COptionsDlg* pThis = (COptionsDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitNotifyOptions(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					// initialize the controls
					pThis->m_hWndNotifyOptions = hWnd;
					return 0;
				}

				case PSN_KILLACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					//Get Notify Options
					if(!pThis->GetNotifyOptions())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:
					return 0;

				case PSN_RESET:
					return 0;
			}
    	}
	}

	return FALSE;   
}


////////////////////////////////////////////////////////////////
// COptionsDlg::ErrorOptionsProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK COptionsDlg::ErrorOptionsProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE * ps;

	switch (message)
	{
		case WM_INITDIALOG:
		{	
			EXC_BEGIN
			// save off the PROPSHEETPAGE information
			ps = (PROPSHEETPAGE*)lParam;
			COptionsDlg* pThis = (COptionsDlg*)SetThis(hWnd, ps->lParam);
			pThis->InitErrorOptions(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					// initialize the controls
					pThis->m_hWndErrorOptions = hWnd;
					return 0;
				}

				case PSN_KILLACTIVE:
				{	
					//Get the "this" pointer
					COptionsDlg* pThis = (COptionsDlg*)GetThis(hWnd);

					//Get Error Options
					if(!pThis->GetErrorOptions())
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:
					return 0;

				case PSN_RESET:
					return 0;
			}
    	}
	}

	return FALSE;   
}


static const ULONG RES_TEXT[]	= { IDT_RESTRICTION1, IDT_RESTRICTION2, IDT_RESTRICTION3, IDT_RESTRICTION4, IDT_RESTRICTION5, IDT_RESTRICTION6, IDT_RESTRICTION7, /*IDT_RESTRICTION8, IDT_RESTRICTION9, IDT_RESTRICTION10, IDT_RESTRICTION11, IDT_RESTRICTION12*/ };
static const ULONG RES_COMBOS[]	= { IDC_RESTRICTION1, IDC_RESTRICTION2, IDC_RESTRICTION3, IDC_RESTRICTION4, IDC_RESTRICTION5, IDC_RESTRICTION6, IDC_RESTRICTION7, /*IDC_RESTRICTION8, IDC_RESTRICTION9, IDC_RESTRICTION10, IDC_RESTRICTION11, IDC_RESTRICTION12*/ };
static const ULONG RES_EDIT[]	= { IDE_RESTRICTION1, IDE_RESTRICTION2, IDE_RESTRICTION3, IDE_RESTRICTION4, IDE_RESTRICTION5, IDE_RESTRICTION6, IDE_RESTRICTION7, /*IDE_RESTRICTION8, IDE_RESTRICTION9, IDE_RESTRICTION10, IDE_RESTRICTION11, IDE_RESTRICTION12*/ };
//////////////////////////////////////////////////////////////////////////////
// GetSchemaName
//
//////////////////////////////////////////////////////////////////////////////
CHAR* GetSchemaName(GUID guidSchema)
{
	for(ULONG i=0; i<NUMELE(g_rgSchemaInfo); i++)
	{
		if(guidSchema == *(g_rgSchemaInfo[i].pguidSchema))
			return g_rgSchemaInfo[i].pszSchemaName;
	}

	return NULL;
}



////////////////////////////////////////////////////////////////
// CSchemaDlg::CSchemaDlg
//
/////////////////////////////////////////////////////////////////
CSchemaDlg::CSchemaDlg(HWND hWnd, HINSTANCE hInst, CMDIChild* pCMDIChild)
	: CDialogBase(hWnd, hInst)
{
	ASSERT(pCMDIChild);
	
	//Data
	m_pCMDIChild = pCMDIChild;
	m_hWndSchemas = NULL;
	m_hWndTreeWnd = NULL;
	m_hWndTreeView = NULL;

	//Rowset - This is for the SchemaTree sheet, it uses its own rowset
	//so it doesn't mess up the users rowset, just to display a tree...
	m_pCTreeRowset = NULL;

	//Saved Dialog Values
	m_fEditing = FALSE;
	m_fActive = TRUE;

	m_cSchemas = 0;
	m_rgSchemas = NULL;
	m_rgRestrictionSupport = NULL;
	
	m_iSchemaIndex = LVM_ERR;
	m_iOldSchemaIndex = LVM_ERR;
	
	m_cRestrictions = 0;
	m_fUseRestrictions = TRUE;
	m_fUseProperties = FALSE;
	m_fShowSupported = TRUE;
}


////////////////////////////////////////////////////////////////
// CSchemaDlg::~CSchemaDlg
//
/////////////////////////////////////////////////////////////////
CSchemaDlg::~CSchemaDlg()
{
	m_hWnd = NULL;
	SAFE_DELETE(m_pCTreeRowset);

	SAFE_FREE(m_rgSchemas);
	SAFE_FREE(m_rgRestrictionSupport);
	FreeVariants(m_cRestrictions, m_rgRestrictions);
}


////////////////////////////////////////////////////////////////
// CSchemaDlg::pCTreeRowset
//
/////////////////////////////////////////////////////////////////
CRowset* const CSchemaDlg::pCTreeRowset()
{
	if(!m_pCTreeRowset)
		m_pCTreeRowset = new CRowset(m_pCMDIChild, new CDataSource(m_pCMDIChild));

	return m_pCTreeRowset;
}

////////////////////////////////////////////////////////////////
// CSchemaDlg::Display
//
/////////////////////////////////////////////////////////////////
ULONG CSchemaDlg::Display()
{
	//Now Display the dialog
	PROPSHEETPAGE psp[2];
	PROPSHEETHEADER psh;

	//Header
	psh.dwSize			= sizeof(PROPSHEETHEADER);
	psh.dwFlags			= PSH_PROPSHEETPAGE | PSH_NOAPPLYNOW;
	psh.hwndParent		= m_hWnd;
	psh.hInstance		= m_hInst;
	psh.pszIcon			= NULL;
	psh.pszCaption		= "IDBSchemaRowset";
	psh.nPages			= NUMELE(psp);
	psh.nStartPage		= 0;
	psh.ppsp			= (LPCPROPSHEETPAGE) &psp;

	//Provider
	psp[0].dwSize		= sizeof(PROPSHEETPAGE);
	psp[0].dwFlags		= PSP_USETITLE;
	psp[0].hInstance	= m_hInst;
	psp[0].pszTemplate	= MAKEINTRESOURCE(IDD_SCHEMAROWSET);
	psp[0].pszIcon		= NULL;
	psp[0].pfnDlgProc	= SchemaRowsetProc;
	psp[0].pszTitle		= "Schemas";
	psp[0].lParam		= (LONG)this;

	//Properties
	psp[1].dwSize		= sizeof(PROPSHEETPAGE);
	psp[1].dwFlags		= PSP_USETITLE;
	psp[1].hInstance	= m_hInst;
	psp[1].pszTemplate	= MAKEINTRESOURCE(IDD_SCHEMATREEVIEW);
	psp[1].pszIcon		= NULL;
	psp[1].pfnDlgProc	= SchemaTreeProc;
	psp[1].pszTitle		= "TreeView";
	psp[1].lParam		= (LONG)this;
	
	//Display the Property Sheets
	return PropertySheet(&psh);
}


////////////////////////////////////////////////////////////////
// HRESULT CSchemaDlg::InitSchemas
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::InitSchemas(HWND hWnd)
{
	HRESULT hr = S_OK;
	m_hWndSchemas = hWnd;
	HWND hWndSchemaCombo = GetDlgItem(hWnd, IDC_SCHEMAS);
	LONG  i,iSel = 0;

	//Default Check UseRestrictions and UseProperties
	CheckDlgButton(hWnd, IDB_USEPROPERTIES,		m_fUseProperties ? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_USERESTRICTIONS,	m_fUseRestrictions ? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDB_SHOWSUPPORTED,		m_fShowSupported ? BST_CHECKED : BST_UNCHECKED);

	//Obtain Supported Schemas
	CListBox* pCListBox = m_pCMDIChild->m_pCListBox;
	CSession* pCSession = m_pCMDIChild->m_pCRowset->m_pCSession;

	//Free any previously obtain schemainfo
	SAFE_FREE(m_rgSchemas);
	SAFE_FREE(m_rgRestrictionSupport);

	//Obtain Supported Schemas
	ASSERT(pCSession->m_pIDBSchemaRowset);
	TESTC(pCListBox->OutputPreMethod("IDBSchemaRowset::GetSchemas(&%d, &0x%08x, &0x%08)", m_cSchemas, m_rgSchemas, m_rgRestrictionSupport));
	XTEST(hWnd, hr = pCSession->m_pIDBSchemaRowset->GetSchemas(&m_cSchemas, &m_rgSchemas, &m_rgRestrictionSupport));
	TESTC(pCListBox->OutputPostMethod(hr, "IDBSchemaRowset::GetSchemas(&%d, &0x%08x, &0x%08)", m_cSchemas, m_rgSchemas, m_rgRestrictionSupport));

	//Fill in Schema Combo
	InitSchemaCombo(m_fShowSupported);
	
	//Fill in Type Combo
	for(i=0; i<NUMELE(RES_TEXT); i++)
	{
		HWND hWndResCombo = GetDlgItem(m_hWndSchemas, RES_COMBOS[i]);
		for(ULONG iType=0; iType<g_cVariantTypes; iType++)
		{
			LONG iSel = SendMessage(hWndResCombo, CB_ADDSTRING,	0, (LPARAM)g_rgVariantTypes[iType].pszName);
			SendMessage(hWndResCombo, CB_SETITEMDATA, iSel, (LPARAM)g_rgVariantTypes[iType].lItem);
		}
	}

	//Try to find DBSCHEMA_TABLES by default (or our saved index)
	if(m_iSchemaIndex == LVM_ERR)
		m_iSchemaIndex = SendMessage(hWndSchemaCombo, CB_FINDSTRINGEXACT, 0, (LPARAM)"DBSCHEMA_TABLES");
	m_iSchemaIndex = SendMessage(hWndSchemaCombo, CB_SETCURSEL, m_iSchemaIndex==CB_ERR ? 0 : m_iSchemaIndex, 0);

CLEANUP:
	return hr;
}
	

////////////////////////////////////////////////////////////////
// HRESULT CSchemaDlg::InitSchemaCombo
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::InitSchemaCombo(BOOL fShowSupported)
{
	CHAR szBuffer[MAX_NAME_LEN+1];
	const SCHEMAINFO* pSchemaInfo = NULL;
	HWND hWndSchemaCombo = GetDlgItem(m_hWndSchemas, IDC_SCHEMAS);
	LONG iSel = 0;

	//Delete all the current items...
	SendMessage(hWndSchemaCombo,	CB_RESETCONTENT,	0, 0);

	//Display the complete list of Schemas (if requested)
	if(!fShowSupported)
	{
		for(ULONG i=0; i<NUMELE(g_rgSchemaInfo); i++)
		{
			pSchemaInfo = &g_rgSchemaInfo[i];
		
			//Now Add it to the ComboBox dialog...
			iSel = SendMessage(hWndSchemaCombo, CB_ADDSTRING,	0, (LPARAM)pSchemaInfo->pszSchemaName);
			SendMessage(hWndSchemaCombo, CB_SETITEMDATA, iSel, (LPARAM)pSchemaInfo);
		}
	}

	//Loop over returned Schemas
	for(ULONG i=0; i<m_cSchemas; i++)
	{
		//Find this Schema in our List so we know the "text" name of it...
		pSchemaInfo = NULL;
		for(ULONG j=0; j<NUMELE(g_rgSchemaInfo); j++)
		{
			if(m_rgSchemas[i] == *g_rgSchemaInfo[j].pguidSchema)
			{
				pSchemaInfo = &g_rgSchemaInfo[j];
				break;
			}
		}

		//There may be a provider specific SchemaGuid, 
		//so just display the Guid value and not a name
		if(pSchemaInfo == NULL)
		{
			WCHAR wszBuffer[MAX_NAME_LEN+1];
			StringFromGUID2(m_rgSchemas[i], wszBuffer, MAX_NAME_LEN);
			ConvertToMBCS(wszBuffer, szBuffer, MAX_NAME_LEN);
		}
		
		//Now Add it to the ComboBox dialog...
		//Only add if were only showing supported, or its not in the known list...
		if(fShowSupported || pSchemaInfo==NULL)
		{
			iSel = SendMessage(hWndSchemaCombo, CB_ADDSTRING,	0, (LPARAM)(pSchemaInfo ? pSchemaInfo->pszSchemaName : szBuffer));
			SendMessage(hWndSchemaCombo, CB_SETITEMDATA, iSel, (LPARAM)((ULONG)pSchemaInfo ? (ULONG)pSchemaInfo : i));
		}
	}
	
	return S_OK;
}


////////////////////////////////////////////////////////////////
// HRESULT CSchemaDlg::GetSelectedSchema
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::GetSelectedSchema(SCHEMAINFO** ppSchemaInfo, ULONG* pdwRestrictions)
{
	ASSERT(ppSchemaInfo);
	SCHEMAINFO* pSchemaInfo = NULL;

	if(pdwRestrictions)
		*pdwRestrictions = 0;

	//User has selected a different Schema
	//We need to update our restirction list...
	HWND hWndSchemaCombo = GetDlgItem(m_hWndSchemas, IDC_SCHEMAS);

	//Get Selected Schema
	m_iSchemaIndex = SendMessage(hWndSchemaCombo, CB_GETCURSEL, 0, 0);
	if(m_iSchemaIndex != CB_ERR)
		pSchemaInfo = (SCHEMAINFO*)SendMessage(hWndSchemaCombo, CB_GETITEMDATA, m_iSchemaIndex, 0);
	
	//Provider Specific Schemas have the Item Data as the index rather than a pointer
	if(pSchemaInfo>=0 && (ULONG)pSchemaInfo<m_cSchemas)
	{
		m_iSchemaIndex = (ULONG)pSchemaInfo;
		pSchemaInfo = NULL;
	}
	
	if(pSchemaInfo)
	{
		//Need to find the corresponding restrictions...
		if(pdwRestrictions)
		{
			m_iSchemaIndex = CB_ERR;
			for(ULONG i=0; i<m_cSchemas; i++)
			{
				if(*pSchemaInfo->pguidSchema == m_rgSchemas[i])
				{
					m_iSchemaIndex = i;
					*pdwRestrictions = m_rgRestrictionSupport[m_iSchemaIndex];
					break;
				}
			}
		}
	}
	//This is a little more difficult, since there must be a provider 
	//specific schema, so we need to build up the info...
	else
	{
		//Clear our old SchemaInfo temp storage
		memset(&m_SchemaInfo, 0, sizeof(SCHEMAINFO));

		if(m_iSchemaIndex>=0 && m_iSchemaIndex<m_cSchemas)
		{
			//Figure our how many restrictions are supported
			//This is required since E_INVALIDARG will be produced if
			//cRestrictions > than the number of max restricitons defined for this Guid
			for(ULONG i=0; i<MAX_RESTRICTIONS; i++)
			{
				if(m_rgRestrictionSupport[m_iSchemaIndex] & (0x00000001 << i))
					m_SchemaInfo.cRestrictions = i+1;
				m_SchemaInfo.rgResInfo[i].iOrdinal = i+1;
			}

			//dwRestrictions
			if(pdwRestrictions)
				*pdwRestrictions = m_rgRestrictionSupport[m_iSchemaIndex];

			//pGuid
			m_SchemaInfo.pguidSchema = &m_rgSchemas[m_iSchemaIndex];
		}

		//Return a pointer to our class temp SchemaInfo
		pSchemaInfo = &m_SchemaInfo;
	}

	*ppSchemaInfo = pSchemaInfo;
	return S_OK;
}



////////////////////////////////////////////////////////////////
// HRESULT CSchemaDlg::ChangeRestrictions
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::ChangeRestrictions()
{
	ULONG i=0; 
	HRESULT hr = S_OK;
	SCHEMAINFO* pSchemaInfo = NULL;
	DWORD dwRestrictions = 0;

	//Get the Selected Schema
	TESTC(hr = GetSelectedSchema(&pSchemaInfo, &dwRestrictions));

	//Display RESTRICTION Info
	for(i=0; i<NUMELE(RES_TEXT); i++)
	{
		HWND hWndResText = GetDlgItem(m_hWndSchemas, RES_TEXT[i]);
		HWND hWndResEdit = GetDlgItem(m_hWndSchemas, RES_EDIT[i]);
		HWND hWndResCombo= GetDlgItem(m_hWndSchemas, RES_COMBOS[i]);

		//Display Restirction Name
		if(i < pSchemaInfo->cRestrictions)
			SendMessage(hWndResText, WM_SETTEXT, 0, (LPARAM)(pSchemaInfo->rgResInfo[i].pszName ? pSchemaInfo->rgResInfo[i].pszName : "Unknown"));
		else
			SendMessage(hWndResText, WM_SETTEXT, 0, (LPARAM)"");
			
		//Default Value
		SendMessage(hWndResEdit, WM_SETTEXT, 0, (LPARAM)"");
			
		//Default Type
		SendMessage(hWndResCombo, CB_SETCURSEL,	0, 0);
		EnableWindow(hWndResText,	dwRestrictions & (0x00000001 << i));
	}

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////////////
// HRESULT CSchemaDlg::GetSelectedRestrictions
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::GetSelectedRestrictions()
{
	WCHAR wszBuffer[MAX_NAME_LEN+1];
	HWND hWndSchemaCombo = GetDlgItem(m_hWndSchemas, IDC_SCHEMAS);
	HRESULT hr = S_OK;
	SCHEMAINFO* pSchemaInfo = NULL;

	//Obtain the Currently selected Schema
	TESTC(hr = GetSelectedSchema(&pSchemaInfo));

	//Properties
	m_fUseProperties = IsDlgButtonChecked(m_hWndSchemas, IDB_USEPROPERTIES);
	m_fShowSupported = IsDlgButtonChecked(m_hWndSchemas, IDB_SHOWSUPPORTED);

	//Free Previous Restrictions
	FreeVariants(m_cRestrictions, m_rgRestrictions);
	m_cRestrictions = 0;
	
	//Obtain Selected Restrictions
	m_fUseRestrictions = IsDlgButtonChecked(m_hWndSchemas, IDB_USERESTRICTIONS);
	if(m_fUseRestrictions)
	{
		m_cRestrictions = 0;
		InitVariants(MAX_RESTRICTIONS, m_rgRestrictions);

		//Get the Restriction info
		for(ULONG i=0; i<NUMELE(RES_TEXT); i++)
		{
			HWND hWndResCombo = GetDlgItem(m_hWndSchemas, RES_COMBOS[i]);
			HWND hWndEditCombo = GetDlgItem(m_hWndSchemas, RES_EDIT[i]);
								
			LONG iSel = SendMessage(hWndResCombo, CB_GETCURSEL, 0, 0);
			DBTYPE wType = (DBTYPE)SendMessage(hWndResCombo, CB_GETITEMDATA, iSel, 0);
				
			//If Not VT_EMPTY we need to setup the restriction Variant
			if(wType != VT_EMPTY)
			{
				wszBuffer[0] = wEOL;
				wSendMessage(hWndEditCombo,	WM_GETTEXT, MAX_NAME_LEN, wszBuffer);
				XTESTC(m_hWnd, hr = StringToVariant(wszBuffer, wType, &m_rgRestrictions[i], CONV_VARBOOL)); 

				//Record we are using this restrictions
				m_cRestrictions = i+1;
			}
		}
	}
			
CLEANUP:
	return hr;
}

	
////////////////////////////////////////////////////////////////
// CSchemaDlg::SchemaRowsetProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CSchemaDlg::SchemaRowsetProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE* ps = NULL;
	
	switch(message)
	{
		case WM_INITDIALOG:
		{
			EXC_BEGIN
			//Save the "this" pointer
			Busy();
			ps = (PROPSHEETPAGE*)lParam;
			CSchemaDlg* pThis = (CSchemaDlg*)SetThis(hWnd, ps->lParam);

			pThis->InitSchemas(hWnd);
			pThis->ChangeRestrictions();

			Busy(OFF);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}

		case WM_COMMAND:
		{
			//CBN_SELCHANGE ListBox Selection change
			switch(GET_WM_COMMAND_CMD(wParam, lParam))
			{
				//Selection change in a list box occurred
				case CBN_SELCHANGE:
				{	
					//See which combo box has changed
					switch(GET_WM_COMMAND_ID(wParam, lParam))
					{
						case IDC_SCHEMAS:
						{
							//Get the "this" pointer
							CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);

							//Get Selected Schema
							pThis->ChangeRestrictions();
							return 0;
						}
					}
					break;
				}
				
				//Selection change in a list box occurred
				case CBN_DROPDOWN:
				{	
					//See which combo box has changed
					switch(GET_WM_COMMAND_ID(wParam, lParam))
					{
						case IDC_SCHEMAS:
						{
							//Get the "this" pointer
							CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
							
							//Update button state
							BOOL fShowSupported = IsDlgButtonChecked(hWnd, IDB_SHOWSUPPORTED);

							//Get Selected Schema
							//Only need to do this if (ShowSupported) has changed...
							if(fShowSupported != pThis->m_fShowSupported)
							{
								pThis->m_fShowSupported = fShowSupported;
								pThis->InitSchemaCombo(fShowSupported);
							}
							return 0;
						}
					}
					break;
				}

				return FALSE;
			}

			switch (GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDB_SETPROPERTIES:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					CMDIChild* pCMDIChild = pThis->m_pCMDIChild;
					CRowset* pCRowset = pCMDIChild->m_pCRowset;
					CDataSource* pCDataSource = pCRowset->m_pCDataSource;
					pThis->m_pCMDIChild->m_pCPropDlg->SetProperties(hWnd, ROWSET, pCRowset->m_pIRowsetInfo, pCDataSource->m_pIDBProperties, &pCMDIChild->m_cPropSets, &pCMDIChild->m_rgPropSets);
					return 0;
				}
			}
			break;

		}//WM_COMMAND

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case PSN_SETACTIVE:
				{	
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					pThis->m_fActive = TRUE;
					return 0;
				}

				case PSN_KILLACTIVE://Switch
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					if(FAILED(pThis->GetSelectedRestrictions()))
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_APPLY:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					CMDIChild* pCMDIChild = pThis->m_pCMDIChild;
					HRESULT hr = S_OK;

					if(pThis->m_fEditing)
					{
						//There is a bug in the TreeView control for editing here is KB article
						//Article ID: Q130691 BUG: ESC/ENTER Keys Don't Work When Editing Labels in TreeView
						//So one way to work around this is to just have a flag (m_fEditing)
						//to indicate we were in editing mode.
						SendMessage(pThis->m_hWndTreeView, TVM_ENDEDITLABELNOW, (WPARAM) (wParam==IDCANCEL ? TRUE : FALSE), (LPARAM)0);
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}

					if(pThis->m_fActive)
					{
						//Setup Properties
						ULONG cPropSets = 0;
						DBPROPSET* rgPropSets = NULL;
						if(pThis->m_fUseProperties)
						{
							cPropSets = pCMDIChild->m_cPropSets;
							rgPropSets = pCMDIChild->m_rgPropSets;
						}

						//Get the selected Schema...
						SCHEMAINFO* pSchemaInfo = NULL;
						TESTC(hr = pThis->GetSelectedSchema(&pSchemaInfo));
						
						//Now Display the Rowset
						TESTC(hr = pCMDIChild->CreateRowset(ROWSET_FROMSCHEMA, NULL, NULL, cPropSets, rgPropSets, IID_IUnknown, pSchemaInfo->pguidSchema, pThis->m_cRestrictions, pThis->m_rgRestrictions));
					}

				CLEANUP:
					Busy(OFF);
					if(FAILED(hr))
					{
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_RESET:		//CANCEL
					return 0;

			}
    	}//WM_NOTIFY

	}//switch(message);

	return FALSE;
}


////////////////////////////////////////////////////////////////
// HRESULT CSchemaDlg::InitTreeView
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::InitTreeView(HWND hWnd)
{
	HRESULT hr = S_OK;
	m_hWndTreeWnd = hWnd;

	//Saved Defaults
	m_fEditing = FALSE;
	m_iOldSchemaIndex = LVM_ERR;
	m_hWndTreeView = GetDlgItem(hWnd, IDC_TREEVIEW);

	//Create the Table ImageList
	HIMAGELIST hTableImageList = ImageList_Create(16, 16, ILC_MASK, 1, 0 );
	//IDI_ROWSETICON - normal rowset icon
	HICON hIcon = LoadIcon(m_hInst, MAKEINTRESOURCE(IDI_ROWSETICON));
	ImageList_AddIcon(hTableImageList, hIcon);
	//IDI_ROW_NORMAL - normal row icon
	hIcon = LoadIcon(m_hInst, MAKEINTRESOURCE(IDI_ROW_NORMAL));
	ImageList_AddIcon(hTableImageList, hIcon);

	//Set image list to the Table Window 
	TreeView_SetImageList(m_hWndTreeView, hTableImageList, TVSIL_NORMAL);

	//Connect using the Same DataSource...
	TESTC(hr = pCTreeRowset()->CreateConnection(m_pCMDIChild->m_pCRowset));

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////////////
// CSchemaDlg::SchemaTreeProc
//
/////////////////////////////////////////////////////////////////
BOOL CALLBACK CSchemaDlg::SchemaTreeProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	static PROPSHEETPAGE* ps = NULL;

	switch(message)
	{
		case WM_INITDIALOG:
		{
			EXC_BEGIN
			//Save the "this" pointer
			ps = (PROPSHEETPAGE*)lParam;
			CSchemaDlg* pThis = (CSchemaDlg*)SetThis(hWnd, ps->lParam);
			
			//Initialize the TreeView Page
			pThis->InitTreeView(hWnd);
			EXC_END_(hWnd, EndDialog(hWnd, FALSE));
			return TRUE;
		}


		case WM_COMMAND:
		{
			//Filter out any Control Notification codes
			if(GET_WM_COMMAND_CMD(wParam, lParam) > 1)
			{
				return UNHANDLED_MSG;
			}

			//Regular command messages
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDMENU_OPENROWSET:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);

					//Send a IDMENU_COPYTEXT to copy the text to the edit box
					SendMessage(hWnd, WM_COMMAND, GET_WM_COMMAND_MPS(IDMENU_COPYTEXT, hWnd, 0));

					//Delete to our OpenRowset dialog box...
					DialogBoxParam(pThis->m_hInst, MAKEINTRESOURCE(IDD_OPENROWSET), hWnd, pThis->m_pCMDIChild->OpenRowsetProc, (LPARAM)pThis->m_pCMDIChild);
					return 0;
				}		  

				case IDMENU_COPYTEXT:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					CHAR szBuffer[MAX_NAME_LEN+1];
					
					//Get the Selected Item
					HTREEITEM hItem = (HTREEITEM)SendMessage(pThis->m_hWndTreeView, TVM_GETNEXTITEM, (WPARAM)TVGN_CARET, (LPARAM)TVI_ROOT);
					
					if(TV_GetItemText(pThis->m_hWndTreeView, hItem, szBuffer, MAX_NAME_LEN))
					{
						//Now just need to place this name in the EditBox
						//Inserted after the current "caret"
						ReplaceEditBoxSelection(pThis->m_pCMDIChild->m_hWndEditBox, szBuffer);
					}
					return 0;
				}		  
			
				case IDB_DISPLAYTREE:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);

					//Make sure we do a regeneration of the tree
					pThis->DisplayTree();
					return 0;
				}
			}
		
			return FALSE;
		}

		// Update status bar to reflect menu selection
		case WM_MENUSELECT:
		{	
			//Get the "this" pointer
			CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
			pThis->m_pCMDIChild->m_pCMainWindow->DisplayStatusBarItem(LOWORD(wParam));
			return 0;
		}

		case WM_CONTEXTMENU:
		{	
			CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
		
			//Only display the Context menu, if there is a selected item
			HTREEITEM hItem = (HTREEITEM)SendMessage(pThis->m_hWndTreeView, TVM_GETNEXTITEM, (WPARAM)TVGN_CARET, (LPARAM)TVI_ROOT);

			if(hItem)
			{
				DisplayContextMenu(	pThis->m_hInst, 
								hWnd,
								IDMENU_TREESCHEMA, 
								LOWORD(lParam),
								HIWORD(lParam),
								hWnd
								);
			}
			return FALSE;
		}

		case WM_NOTIFY:
		{	
			switch (((NMHDR*)lParam)->code) 
    		{
				case TVN_BEGINLABELEDIT:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);

					//Idicate we have started to edit
					pThis->m_fEditing = TRUE;
					return FALSE; //Allow the edited change
				}

				case TVN_ENDLABELEDIT:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					pThis->m_fEditing = FALSE;
					return TRUE; //Allow the edited change
				}

				case TVN_SELCHANGED:
				{
					//There is a problem with the SELCHANGED notification
					//It can be sent when either a item is selected or
					//DELETED, since when an item deleted the selection moves
					//to a different selection.
					NM_TREEVIEW* pTreeView = (NM_TREEVIEW*)lParam;
					if(pTreeView->itemNew.state & TVIS_SELECTED && pTreeView->action)
					{
						//Get the "this" pointer
						CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
						pThis->RefreshTreeControls();
					}
					return 0;
				}

				case TTN_GETDISPINFO:
				{	
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					return 0;
				}
				
				case PSN_SETACTIVE:
				{	
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
			
					//Need to display the specified SchemaRowset
					pThis->RefreshTreeControls();
					pThis->m_fActive = FALSE;
					return 0;
				}

				case PSN_KILLACTIVE://Switch
					return 0;

				case PSN_APPLY:
				{
					//Get the "this" pointer
					CSchemaDlg* pThis = (CSchemaDlg*)GetThis(hWnd);
					if(pThis->m_fEditing)
					{
						//There is a bug in the TreeView control for editing here is KB article
						//Article ID: Q130691 BUG: ESC/ENTER Keys Don't Work When Editing Labels in TreeView
						//So one way to work around this is to just have a flag (m_fEditing)
						//to indicate we were in editing mode.
						SendMessage(pThis->m_hWndTreeView, TVM_ENDEDITLABELNOW, (WPARAM) (wParam==IDCANCEL ? TRUE : FALSE), (LPARAM)0);
						SetWindowLong(hWnd, DWL_MSGRESULT, PSNRET_INVALID_NOCHANGEPAGE);
						return TRUE;
					}
					return 0;
				}

				case PSN_RESET:		//CANCEL
					return 0;

			}
    	}//WM_NOTIFY

	}//switch(message);

	return FALSE;
}


////////////////////////////////////////////////////////////////
// HRESULT CSchemaDlg::RefreshTreeControls
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::RefreshTreeControls()
{
	HWND hWndColName = GetDlgItem(m_hWndTreeWnd, IDT_COLUMNNAME);

	if(m_iSchemaIndex != m_iOldSchemaIndex)
	{
		EnableWindow(m_hWndTreeView, FALSE);
		EnableWindow(hWndColName, FALSE);
		EnableWindow(GetDlgItem(m_hWndTreeWnd, IDMENU_COPYTEXT), FALSE);
		EnableWindow(GetDlgItem(m_hWndTreeWnd, IDMENU_OPENROWSET), FALSE);
	}
	else
	{
		EnableWindow(m_hWndTreeView, TRUE);
		EnableWindow(hWndColName, TRUE);

		//Get the Selected TreeItem
		HTREEITEM hItem = (HTREEITEM)SendMessage(m_hWndTreeView, TVM_GETNEXTITEM, (WPARAM)TVGN_CARET, (LPARAM)TVI_ROOT);

		//Update the ColName header...
		LONG iOrdinal = TV_GetItemParam(m_hWndTreeView, hItem);
		if(hItem && iOrdinal != LVM_ERR && iOrdinal<(LONG)pCTreeRowset()->m_cColumns)
		{
			wSendMessage(hWndColName, WM_SETTEXT, 0, pCTreeRowset()->GetColName(iOrdinal));
			EnableWindow(GetDlgItem(m_hWndTreeWnd, IDMENU_COPYTEXT), TRUE);
			EnableWindow(GetDlgItem(m_hWndTreeWnd, IDMENU_OPENROWSET), TRUE);
		}
		else
		{
			SendMessage(hWndColName, WM_SETTEXT, 0, (LPARAM)"");
			EnableWindow(GetDlgItem(m_hWndTreeWnd, IDMENU_COPYTEXT), FALSE);
			EnableWindow(GetDlgItem(m_hWndTreeWnd, IDMENU_OPENROWSET), FALSE);
		}
	}

	return S_OK;
}


////////////////////////////////////////////////////////////////
// CSchemaDlg::DisplayTree
//
/////////////////////////////////////////////////////////////////
HRESULT CSchemaDlg::DisplayTree()
{
	Busy();
	HRESULT hr = E_FAIL;
	LONG cRowsAffected = DB_COUNTUNAVAILABLE;

	ULONG i,cRowsObtained = 0;
	HROW* rghRows = NULL;
	CHAR szBuffer[MAX_COL_SIZE];
	
	ULONG cColOrds = 0;
	ULONG* rgColOrds = NULL;

	ULONG iOrdinal, iColumn;
	HTREEITEM hFoundItem, hParent;
	DBCOLUMNINFO* pColInfo = NULL;
	DBBINDING* pBinding = NULL;
	COptionsDlg* pCOptionsDlg = m_pCMDIChild->GetOptionsObj();
	void* pData = NULL;

	//Setup Properties
	ULONG cPropSets = 0;
	DBPROPSET* rgPropSets = NULL;
	SCHEMAINFO* pSchemaInfo = NULL;
	if(m_fUseProperties)
	{
		cPropSets = m_pCMDIChild->m_cPropSets;
		rgPropSets = m_pCMDIChild->m_rgPropSets;
	}

	//Get the Selected Schema
	TESTC(hr = GetSelectedSchema(&pSchemaInfo));

	//GetRowset, (either from Command, IOpenRowset, or Schema's
	TESTC(hr = pCTreeRowset()->CreateRowset(ROWSET_FROMSCHEMA, cPropSets, rgPropSets, &cRowsAffected, NULL, NULL, IID_IUnknown, pSchemaInfo->pguidSchema, m_cRestrictions, m_rgRestrictions));

	// Empty IRowset
	if(pCTreeRowset()->m_pIUnknown == NULL)
		goto CLEANUP;

	//Clear TreeView object
	SendMessage(m_hWndTreeView, TVM_DELETEITEM, 0, (LPARAM)TVI_ROOT);

	//Create ColumnInfo
	TESTC(hr = pCTreeRowset()->GetColInfo());

	///Create Accessors and Setup bindings, (don't bind bookmark)
	TESTC(hr = pCTreeRowset()->CreateAccessors(FALSE));

	//All columns that are restrictions are done first, this creates the tree
	//nodes.  Then all remaining columns are just elements under the last node.
	//We need to setup this array of ordinals to use to indicate the ordering...
	SAFE_ALLOC(rgColOrds, ULONG, pCTreeRowset()->m_cColumns);

	//Add all restriction columns first
	cColOrds = pSchemaInfo->cRestrictions;
	for(i=0; i<cColOrds; i++)
	{
		ASSERT(pSchemaInfo->rgResInfo[i].iOrdinal < pCTreeRowset()->m_cColumns);
		rgColOrds[i] = pSchemaInfo->rgResInfo[i].iOrdinal;
	}
	
	//Add all remaining columns, (that are not already in the list...)
	iColumn = cColOrds;
	for(i=0; i<pCTreeRowset()->m_cBindings; i++)
	{
		BOOL bFound = FALSE;
		iOrdinal = pCTreeRowset()->m_rgBindings[i].iOrdinal;
		for(ULONG j=0; j<cColOrds && !bFound; j++)
		{
			if(iOrdinal == rgColOrds[j])
				bFound = TRUE;
		}

		//Add it to the list, if it doesn't already exist...
		if(!bFound)
		{
			rgColOrds[iColumn] = iOrdinal;	
			iColumn++;
		}
	}

	//Grab all the rows
	pData = pCTreeRowset()->m_pData;
	while(TRUE)
	{
		TESTC(hr = pCTreeRowset()->GetNextRows(0, 1, &cRowsObtained, &rghRows));
		
		//ENDOFROWSET
		if(cRowsObtained==0)
			break;
	
		//Get the Data
		TESTC(hr = pCTreeRowset()->GetData(rghRows[0]));

		//Loop through all the restrictions first
		//We will create nodes based upon the sort order of the restrictions.
		//Any column that is not in the sort oder will just be an item under a node
		hParent = NULL;
		for(ULONG i=0; i<cColOrds; i++)
		{
			//Get the Data for this Column
			pBinding = m_pCTreeRowset->GetBinding(rgColOrds[i]);
			pColInfo = m_pCTreeRowset->GetColInfo(rgColOrds[i]);
			TESTC(hr = m_pCTreeRowset->GetColumnData(pBinding, pData, NULL, NULL, NULL, szBuffer, MAX_COL_SIZE, pCOptionsDlg->m_dwConvFlags, pColInfo->wType));

			//Insert it into the TreeView
			if(szBuffer[0])
			{
				//Make sure it doesn't already exist...
				hFoundItem = TV_FindItem(m_hWndTreeView, hParent, szBuffer);
				if(hFoundItem)
				{
					hParent = hFoundItem;
				}
				else
				{
					//Insert the Item (using node Icon - 0)
					hParent = TV_InsertItem(m_hWndTreeView, hParent, TVI_LAST, szBuffer, rgColOrds[i], 0, 0);
				}
			}
		}

		//Now go through the regular non-node items and just 
		//add the the latest parent
		for(i=cColOrds; i<pCTreeRowset()->m_cBindings; i++)
		{
			pBinding = m_pCTreeRowset->GetBinding(rgColOrds[i]);
			pColInfo = m_pCTreeRowset->GetColInfo(rgColOrds[i]);

			//Get the Data for this Column
			TESTC(hr = m_pCTreeRowset->GetColumnData(pBinding, pData, NULL, NULL, NULL, szBuffer, MAX_COL_SIZE, pCOptionsDlg->m_dwConvFlags, pColInfo->wType));

			//Insert it into the TreeView
			if(szBuffer[0])
			{
				//Display the Item (using item icon - 1)
				TV_InsertItem(m_hWndTreeView, hParent, TVI_LAST, szBuffer, rgColOrds[i], 1, 1);
			}
		}

		//Release the rows obtained
		TESTC(hr = pCTreeRowset()->ReleaseRows(cRowsObtained, rghRows));
	}

	//We now have displayed this tree
	m_iOldSchemaIndex = m_iSchemaIndex;

CLEANUP:
	SAFE_FREE(rghRows);
	SAFE_FREE(rgColOrds);
	RefreshTreeControls();
	Busy(OFF);
	return hr;
}


